#include "display.h"

SDL_Window   * SDLWindow = NULL;
SDL_Surface  * SDLSurface;

static int SDL_BPP = 0;
static int SDL_Width = 0;
static int SDL_Height = 0;

static int XPos=0,YPos=0;
static int XMax=0,YMax=0;

struct gstruct {
	int active;
	unsigned int *map;
	double gamma;
	double gain;
	int brightness;
	int src_bpp;
	};

#define MAX_PROPERTIES 10
static struct gstruct PropList[MAX_PROPERTIES];

static unsigned int * GetPropertyMap(int p, int src_bpp);
static unsigned int * CreatePropertyMap_16bpp_565(int p);
static unsigned int * CreatePropertyMap_16bpp_888(int p);
static unsigned int * CreatePropertyMap_8bpp_888(int p);
static unsigned int * CreatePropertyMap_8bpp_565(int p);

static void DrawBox(int x1, int y1, int x2, int y2, int val,unsigned int *map);
static int _SDLDisplay_8bpp(struct Image *img, unsigned int *map, int x_off, int y_off);
static int _SDLDisplay_16bpp(struct Image *img, unsigned int *map, int x_off, int y_off);
static int _SDLDisplay_24bpp(struct Image *img, unsigned int *map, int x_off, int y_off);

void
Message(const char *str)
	{
	fprintf(stderr,"%s\n",str);
	fflush(stderr);
	}

void
FatalError(const char *str)
	{
	Message(str);
	exit(1);
	}

// Create a "property", ie a collection of display parameters to be applied to
// an image
int
CreateProperty(int brightness, double gain, double gamma)
	{
	int i;

	for(i=0; i<MAX_PROPERTIES; ++i) {

	   if (! PropList[i].active) {
		PropList[i].active = 1;
		PropList[i].src_bpp = 0;
		PropList[i].gain = gain;
		PropList[i].gamma = gamma;
		PropList[i].brightness = brightness;
		PropList[i].map = NULL;
		return i;
		}
	   }

	printf("CreateProperty: no spare tables\n");
	fflush(stdout);

	// no spare tables
	return -1;
	}

void
SetProperty_brightness(int p, int brightness)
	{
	if (brightness != PropList[p].brightness) {
	   PropList[p].brightness = brightness;
	   if (PropList[p].map) Free(PropList[p].map);
	   PropList[p].map = NULL;
	   }
	}

void
SetProperty_gamma(int p, double gamma)
	{
	if (gamma != PropList[p].gamma) {
	   PropList[p].gamma = gamma;
	   if (PropList[p].map) Free(PropList[p].map);
	   PropList[p].map = NULL;
	   }
	}

void
SetProperty_gain(int p, double gain)
	{

	if (gain != PropList[p].gain) {
	   PropList[p].gain = gain;
	   if (PropList[p].map) Free(PropList[p].map);
	   PropList[p].map = NULL;
	   }
	}

int
AdjustProperty_brightness(int p, int brightness)
	{
	int b = PropList[p].brightness + brightness;

	if (b<0) b=0;

	if (b != PropList[p].brightness) {
	   PropList[p].brightness = b;
	   if (PropList[p].map) Free(PropList[p].map);
	   PropList[p].map = NULL;
	   }

	return b;
	}

double
AdjustProperty_gamma(int p, double gamma)
	{
	double g = PropList[p].gamma + gamma;

	if (g<0) g=0;

	if (g != PropList[p].gamma) {
	   PropList[p].gamma = g;
	   if (PropList[p].map) Free(PropList[p].map);
	   PropList[p].map = NULL;
	   }

	return g;
	}

double
AdjustProperty_gain(int p, double gain)
	{
	double g = PropList[p].gain + gain;

	if (g<0) g=0;

	if (g != PropList[p].gain) {
	   PropList[p].gain = g;
	   if (PropList[p].map) Free(PropList[p].map);
	   PropList[p].map = NULL;
	   }

	return g;
	}

void
DeleteProperty(int i)
	{
	if (i<0 || i >= MAX_PROPERTIES) return;

	PropList[i].gain = 1.0;
	PropList[i].gamma = 1.0;
	if (PropList[i].map) Free(PropList[i].map);
	PropList[i].map = NULL;
	PropList[i].src_bpp = 0;
	PropList[i].active = 0;
	}
	
// Generate a lookup table that will convert image values of the designated src_bpp into pixels
// taking into account the property adjustments
static unsigned int *
GetPropertyMap(int p, int src_bpp)
	{

	// Has someone already generated the map ?
	if (PropList[p].map != NULL && PropList[p].src_bpp == src_bpp)
	   return PropList[p].map;

	// Otherwise create a lookup table that maps image pixels + display properties
	// onto screen pixels.

	switch (src_bpp) {
	   case 8:
		if (SDL_BPP==16) return CreatePropertyMap_8bpp_565(p);
		if (SDL_BPP==32) return CreatePropertyMap_8bpp_888(p);
		break;
	   case 16:
		if (SDL_BPP==16) return CreatePropertyMap_16bpp_565(p);
		if (SDL_BPP==32) return CreatePropertyMap_16bpp_888(p);
		printf("Unknown SDL_BPP %d\n",SDL_BPP);
		exit(1);
		break;
	   default:
		printf("unknown or not supported src_bpp %d\n",src_bpp);
		exit(1);
		break;
	   }

	printf("GetPropertyMap: don't support bpp=%d and SDL_BPP=%d\n",src_bpp,SDL_BPP);
	exit(1); return NULL;
	}

/*
 * create a lookup table that maps a value (multiplied by gain) to 565
 */
static unsigned int *
CreatePropertyMap_16bpp_565(int p)
        {
        int i;
	int brightness = PropList[p].brightness;
	double gain = PropList[p].gain;
	double gamma = PropList[p].gamma;
	unsigned int *map;

	if (PropList[p].map == NULL) 
	   PropList[p].map = (unsigned int *)Malloc(65536 * 4);

	map = PropList[p].map;

        for(i=0; i<65536; ++i) {
	   int n = i * gain + brightness;
	   if (n>65535) n = 65535;
	   if (n<0) n=0;

           unsigned int v = (unsigned int) (pow((double)n / 65535.0, 1.0/gamma) * 65535);
           if (v>65535) v=65535;
	   if (v<0) v=0;

	   // convert to 5-6-5 format
           map[i] = (v&0xf800) | ((v&0xfc00)>>5) | (v>>11);
           }

	PropList[p].src_bpp = 16;
        return map;
        }

static unsigned int *
CreatePropertyMap_16bpp_888(int p)
        {
        int i;
	int brightness = PropList[p].brightness;
	double gain = PropList[p].gain;
	double gamma = PropList[p].gamma;
	unsigned int *map;

	if (PropList[p].map == NULL) 
	   PropList[p].map = (unsigned int *)Malloc(65536 * 4);

	map = PropList[p].map;

        for(i=0; i<65536; ++i) {
	   int n = i * gain + brightness;
	   if (n>65535) n = 65535;
	   if (n<0) n=0;

           unsigned int v = (unsigned int) (pow((double)n / 65535.0, 1.0/gamma) * 65535);
           if (v>65535) v=65535;
	   if (v<0) v=0;

	   v >>= 8;

	   // convert to 5-6-5 format
           map[i] = v | (v<<8) | (v<<16);
           }

	PropList[p].src_bpp = 16;
        return map;
        }

static unsigned int *
CreatePropertyMap_8bpp_565(int p)
        {
        int i;
	int brightness = PropList[p].brightness;
	double gain = PropList[p].gain;
	double gamma = PropList[p].gamma;
	unsigned int *map;

	if (PropList[p].map == NULL) 
	   PropList[p].map = (unsigned int *)Malloc(256 * 4);

	map = PropList[p].map;

        for(i=0; i<256; ++i) {
	   int n = i * gain + brightness;
	   if (n>255) n = 255;
	   if (n<0) n=0;

           unsigned int v = (unsigned int) (pow((double)n / 255.0, 1.0/gamma) * 255);
           if (v>255) v=255;
	   if (v<0) v=0;

	   // convert to 8-8-8 format
	   unsigned int v5 = v >> 3;
	   unsigned int v6 = v >> 2;

           map[i] = v5 | (v6<<5) | (v5<<11);
           }

	PropList[p].src_bpp = 8;
        return map;
        }

static unsigned int *
CreatePropertyMap_8bpp_888(int p)
        {
        int i;
	int brightness = PropList[p].brightness;
	double gain = PropList[p].gain;
	double gamma = PropList[p].gamma;
	unsigned int *map;

	if (PropList[p].map == NULL) 
	   PropList[p].map = (unsigned int *)Malloc(256 * 4);

	map = PropList[p].map;

        for(i=0; i<256; ++i) {
	   int n = i * gain + brightness;
	   if (n>255) n = 255;
	   if (n<0) n=0;

           unsigned int v = (unsigned int) (pow((double)n / 255.0, 1.0/gamma) * 255);
           if (v>255) v=255;
	   if (v<0) v=0;

	   // convert to 8-8-8 format
           map[i] = v | (v<<8) | (v<<16);
           }

	PropList[p].src_bpp = 8;
        return map;
        }
int
InitProperties()
	{
	int i;

	for(i=0; i<MAX_PROPERTIES; ++i) {
	   PropList[i].active = 0;
	   PropList[i].brightness = 0;
	   PropList[i].gamma = 1.0;
	   PropList[i].gain = 1.0;
	   PropList[i].src_bpp = 0;
	   PropList[i].map = NULL;
	   }

	return 0;
	}

int
GetDisplayWidth()
	{
	return SDL_Width; 
	}

int 
GetDisplayHeight()
	{
	return SDL_Height;
	}

int
InitDisplay(char *caption, int width, int height, int bpp)
	{
	unsigned int sdlflags;
	int i,sdlbpp;

	if (SDL_Init(SDL_INIT_VIDEO) == -1) {
	   FatalError("Could not init SDL\n");
	   exit(1);
	   }

	SDL_SetHint(SDL_HINT_RENDER_SCALE_QUALITY, "linear");

	//printf("Asking for %d %d\n",width,height); fflush(stdout);

	SDLWindow = SDL_CreateWindow(
		"ninox",
		SDL_WINDOWPOS_UNDEFINED,
		SDL_WINDOWPOS_UNDEFINED,
		width, height, 
		SDL_WINDOW_SHOWN | SDL_WINDOW_RESIZABLE
		);

 	if (SDLWindow == NULL) {
    	   FatalError(SDL_GetError());
    	   return(0);
  	   }

	SDLSurface = SDL_GetWindowSurface(SDLWindow);

	SDL_ShowCursor(1);
	SDL_SetWindowTitle(SDLWindow,caption);
	//SDL_EnableKeyRepeat(50,100);

	SDL_BPP = 32;
	SDL_Width = width;
	SDL_Height = height;

	SDL_GetWindowSize(SDLWindow,&XMax,&YMax);
	SDL_GetWindowPosition(SDLWindow,&XPos,&YPos);
	return(1);
	}

static void
DrawBox(int x1, int y1, int x2, int y2, int val, unsigned int *map)
	{
	int x,y;
	const SDL_PixelFormat *f = SDLSurface->format;
	int bpp = f->BytesPerPixel;

	SDL_LockSurface(SDLSurface);
	if (SDL_BPP == 32) {
    	   register unsigned int *dst1 = (unsigned int *)((Uint8 *)SDLSurface->pixels + y1 * SDLSurface->pitch + x1 * bpp);
    	   register unsigned int *dst2 = (unsigned int *)((Uint8 *)SDLSurface->pixels + y1 * SDLSurface->pitch + x1 * bpp);
	   for(x=x1; x<x2; ++x) { *(dst1++) = (unsigned int)map[val]; *(dst2++) = (unsigned int)map[val]; }
	   //
	   // left and right sides
    	   dst1 = (unsigned int *)((Uint8 *)SDLSurface->pixels + y1 * SDLSurface->pitch + x1 * bpp);
    	   dst2 = (unsigned int *)((Uint8 *)SDLSurface->pixels + y1 * SDLSurface->pitch + x2 * bpp);
	   for(y=y1; y<y2; ++y) { *dst1 = (unsigned int)map[val]; *dst2 = (unsigned int)map[val]; 
		dst1 += SDLSurface->pitch; dst2 += SDLSurface->pitch;
		}
	   }
	else {
	   printf("DrawBox: Unsupported depth %d\n",SDL_BPP);
	   exit(1);
	   }

	SDL_UnlockSurface(SDLSurface);
	SDL_UpdateWindowSurface(SDLWindow);
	}

void
ShutdownDisplay()
	{
	SDL_Quit();
	}

// Display an image, format can be: 8bpp mono, 16bpp mono, 24bpp colour
// p is a property index
// (x_off,y_off) is the origin of the image on the window, in case the image is smaller
// than the window
// return 1 for success, 0 on error

int
DisplayImage(struct Image *img, int p, int x_off,int y_off)
	{
	unsigned int *map = NULL;
	int rval;
	int x,y;

	if (img->depth != 24 && !(map = GetPropertyMap(p,img->depth))) {
	   printf("DisplayImage: cannot get property map\n");
	   return 0;
	   }

	SDL_GetWindowPosition(SDLWindow,&XPos,&YPos);
	SDL_GetWindowSize(SDLWindow,&XMax,&YMax);

	//Print("Window at (%d %d)  size (%d %d)\n",XPos,YPos,XMax,YMax); fflush(stdout);

	switch(img->depth) {
	   case 8: rval= _SDLDisplay_8bpp(img,map,x_off,y_off); break;
	   case 16: rval= _SDLDisplay_16bpp(img,map,x_off,y_off); break;
	   case 24: rval= _SDLDisplay_24bpp(img,map,x_off,y_off); break;
	   default: printf("DisplayImage: unknown image depth %d\n",img->depth);
		    exit(1); break;
	   }

	return rval;
	}
	
// display an image using a lookup table to convert src values into pixels
// assuming a 8bpp monochrome src image
static int
_SDLDisplay_8bpp(struct Image *img, unsigned int *map, int x_off, int y_off)
	{
	unsigned char *data = (unsigned char *)img->data;
	int x,y,x1,y1,x2,y2,yr;
	int W,width = img->width;
	int H,height = img->height;
	const SDL_PixelFormat *f = SDLSurface->format;
	int bpp = f->BytesPerPixel;

	// may be displaying a subregion
	x1 = img->cutout.x1; y1 = img->cutout.y1; x2 = img->cutout.x2; y2 = img->cutout.y2;

	if (x1<0) x1=0; if (x2>=width) x2=width-1;
	if (y1<0) y1=0; if (y2>= height) y2=height-1;

	W = x2-x1+1;	// dimensions of region we are transferring
	H = y2-y1+1;

	SDL_LockSurface(SDLSurface);

	for(y=y1,yr=0; y<=y2 && yr < YMax; ++y,++yr) {
	   register unsigned char *sptr = data + y * width + x1;

	   if (SDL_BPP == 16) {
    	   	register unsigned short *dst = 
			(unsigned short *)((Uint8 *)SDLSurface->pixels + 
			(y_off+y-y1) * SDLSurface->pitch + x_off * bpp);
		for(x=0; x<W && x<XMax; ++x) *(dst++) = (unsigned short)map[*(sptr++)];
		}
	   if (SDL_BPP == 32) {
    	   	register unsigned int *dst = 
			(unsigned int *)((Uint8 *)SDLSurface->pixels + 
			(y_off+y-y1) * SDLSurface->pitch + x_off * bpp);
	   	for(x=0; x<W && x<XMax; ++x) *(dst++) = map[*(sptr++)];
		}
	   }

	if (0 && ImageTracking && Sub_x && Sub_y) {
	   x1 = Sub_x - ImageTracking/2 - x1; if (x1<1) x1=1;
	   y1 = Sub_y - ImageTracking/2 - y1; if (y1<1) y1=1;
	   x2 = x1 + ImageTracking; if (x2 >= x1+W) { x2 = x1+W-1; }
	   y2 = y1 + ImageTracking; if (y2 >= y1+H) { y2 = y1+H-1; }
	   DrawBox(x1,y1,x2,y2,255,map);
	   }

	SDL_UnlockSurface(SDLSurface);
	SDL_UpdateWindowSurface(SDLWindow);
	}

// display an image using a lookup table to convert src values into pixels
// assuming a 16bpp monochrome src image
static int
_SDLDisplay_16bpp(struct Image *img, unsigned int *map, int x_off, int y_off)
	{
	unsigned short *data = (unsigned short *)img->data;
	int x,y,x1,y1,x2,y2,yr;
	int W,width = img->width;
	int H,height = img->height;
	const SDL_PixelFormat *f = SDLSurface->format;
	int bpp = f->BytesPerPixel;

	// may be displaying a subregion
	x1 = img->cutout.x1; y1 = img->cutout.y1; x2 = img->cutout.x2; y2 = img->cutout.y2;

        //y1 = (img->height - img->dst_height)/2; y2 = y1 + img->dst_height-1;
        //x1 = (img->width - img->dst_width)/2; x2 = x1 + img->dst_width-1;

	if (x1<0) x1=0; if (x2>=width) x2=width-1;
	if (y1<0) y1=0; if (y2>= height) y2=height-1;

	W = x2-x1+1;	// dimensions of region we are transferring
	H = y2-y1+1;

	SDL_LockSurface(SDLSurface);

	for(yr=0,y=y1; y<=y2 && yr<YMax; ++y,++yr) {
	   register unsigned short *sptr = data + y * width + x1;

	   if (SDL_BPP == 16) {
    	   	register unsigned short *dst = 
			(unsigned short *)((Uint8 *)SDLSurface->pixels + 
			(y_off+y-y1) * SDLSurface->pitch + x_off * bpp);
		for(x=0; x<W && x<XMax; ++x) *(dst++) = (unsigned short)map[*(sptr++)];
		}
	   if (SDL_BPP == 32) {
    	   	register unsigned int *dst = 
			(unsigned int *)((Uint8 *)SDLSurface->pixels + 
			(y_off+y-y1) * SDLSurface->pitch + x_off * bpp);
	   	for(x=0; x<W && x<XMax; ++x) *(dst++) = map[*(sptr++)];
		}
	   }

	if (0 && ImageTracking && Sub_x && Sub_y) {
	   x1 = Sub_x - ImageTracking/2 - x1; if (x1<1) x1=1;
	   y1 = Sub_y - ImageTracking/2 - y1; if (y1<1) y1=1;
	   x2 = x1 + ImageTracking; if (x2 >= x1+W) { x2 = x1+W-1; }
	   y2 = y1 + ImageTracking; if (y2 >= y1+H) { y2 = y1+H-1; }
	   DrawBox(x1,y1,x2,y2,255*255,map);
	   }

	SDL_UnlockSurface(SDLSurface);
	SDL_UpdateWindowSurface(SDLWindow);
	}

// display an image using a lookup table to convert src values into pixels
// assuming a 24bpp 8/8/8 RGB src image
static int
_SDLDisplay_24bpp(struct Image *img, unsigned int *map, int x_off, int y_off)
	{
	unsigned char *data = (unsigned char *)img->data;
	int x,y,x1,y1,x2,y2,yr;
	int W,width = img->width;
	int H,height = img->height;
	const SDL_PixelFormat *f = SDLSurface->format;
	int bpp = f->BytesPerPixel;

	// may be displaying a subregion
	x1 = img->cutout.x1; y1 = img->cutout.y1; x2 = img->cutout.x2; y2 = img->cutout.y2;

        //y1 = (img->height - img->dst_height)/2; y2 = y1 + img->dst_height-1;
        //x1 = (img->width - img->dst_width)/2; x2 = x1 + img->dst_width-1;

	if (x1<0) x1=0; if (x2>=width) x2=width-1;
	if (y1<0) y1=0; if (y2>= height) y2=height-1;

	W = x2-x1+1;	// dimensions of region we are transferring
	H = y2-y1+1;

	SDL_LockSurface(SDLSurface);

	for(y=y1,yr=0; y<=y2 && yr<YMax; ++y,++yr) {
	   register unsigned char *sptr = data + (y * width + x1) * 3;

	   if (SDL_BPP == 16) {
    	   	register unsigned short *dst = 
			(unsigned short *)((Uint8 *)SDLSurface->pixels + 
			(y_off+y-y1) * SDLSurface->pitch + x_off * bpp);
		for(x=0; x<W && x<XMax; ++x) {
		   unsigned int v = *(sptr++); 
		   v <<= 8; v |= *(sptr++);
		   v <<= 8; v |= *(sptr++);
		   if (map) *(dst++) = (unsigned short)map[v]; else *(dst++) = v;
		   }
		}
	   if (SDL_BPP == 32) {
    	   	register unsigned int *dst = 
			(unsigned int *)((Uint8 *)SDLSurface->pixels + 
			(y_off+y-y1) * SDLSurface->pitch + x_off * bpp);
	   	for(x=0; x<W && x<XMax; ++x) {
		   unsigned int v = *(sptr++); 
		   v <<= 8; v |= *(sptr++);
		   v <<= 8; v |= *(sptr++);
		   if (map) *(dst++) = (unsigned int)map[v]; else *(dst++) = v;
		   }
		}
	   }

	SDL_UnlockSurface(SDLSurface);
	SDL_UpdateWindowSurface(SDLWindow);
	}

// To be called before the real SDL_Init()
int
GetDesktopSize(int *dw, int *dh)
	{
	int i;
	SDL_DisplayMode current;

	*dw = *dh = 0;

	SDL_Init(SDL_INIT_VIDEO);

	// Get current display mode of all displays.
	for(i = 0; i < SDL_GetNumVideoDisplays(); ++i){

	   int should_be_zero = SDL_GetCurrentDisplayMode(i, &current);

	   if(should_be_zero != 0) {
	   	// In case of error...
	   	SDL_Log("Could not get display mode for video display #%d: %s", i, SDL_GetError());
		return 0;
	   	}
	   else
		{
	   	// On success, print the current display mode.
	   	//SDL_Log("Display #%d: current display mode is %dx%dpx @ %dhz. \n", i, current.w, current.h, current.refresh_rate);
		if (current.w > *dw || current.h > *dh) {
		   *dw = current.w;
		   *dh = current.h;
		   }
		}
	   }

	// Clean up and exit the program.
	SDL_Quit();
	return 1;
	}

