//
// memory.c
//
//	Implements a range of memory allocation & manipulation routines
//	to replace the standard system version.
//
//	The versions here all have some guard checking to catch overruns
//
//	Routines have the same name as their system counterparts but capitalize
//	the first letter (eg Malloc, Free, Strcpy etc).
//
//	They are not interchangeable with the system versions.
//
//	Memory layout:
//
//	Memory is allocated using the system malloc() and realloc(). We must ask for
//	an extra 16 bytes to be guaranteed space for the header and guard values. The
//	layout below starts at the beginning of the memory returned from malloc() or realloc().
//
//	
//	4-byte magic signature 0xabbad0ee
//	4-byte unsigned int size of user-available memory
//	4 x guard indicator bytes  (g1)
//		--- user allocated memory ---
//	4 x guard indicator bytes   (g2)
//
//	Routines that manipulate this memory can do extra validation to check for buffer overruns
//	or corruption of the guard bytes.

#include "ninox.h"

typedef struct  __attribute__ ((__packed__)) {
	unsigned int magic;	// Magic value
	unsigned int sz;	// User available size
	unsigned int g1;	// 4 bytes of guard data
	unsigned char data[1];	// User data, variable size
	} mtype;

#define MAGIC 0xabbad0ee
#define GUARD 0x45454545

static mtype * GetMtypeFromPtr(unsigned char *ptr);
static int CheckSignature(mtype *m);

static mtype *
GetMtypeFromPtr(unsigned char *ptr)
	{
	mtype *m = (mtype *)(ptr - 12);
	return m;
	}

static int
isMtype(mtype *m)
	{
	if (m->magic == MAGIC) return 1;
	else return 0;
	}
	
static int
CheckSignature(mtype *m)
	{
	unsigned char *g2 = ((unsigned char *)m) + 12 + m->sz;

	if (! isMtype(m)) {
	   printf("CheckSignature: pointer missing magic signature\n");
	   fflush(stdout);
	   return 0;
	   }

	// Check the leading guard bytes
	if (m->g1 != GUARD) {
	   printf("CheckSignature: leading GUARD bytes altered - possible corruption: %04x\n",m->g1);
	   fflush(stdout);
	   return 0;
	   }

	// Check the trailing guard bytes
	if (*g2 != GUARD) {
	   printf("CheckSignature: trailing GUARD bytes altered - possible corruption: %04x\n",*g2);
	   fflush(stdout);
	   return 0;
	   }

	// All OK
	return 1;
	}

#ifdef MEMORY_C

void *
Malloc(int nbytes)
	{
	unsigned char *ptr = malloc(nbytes + 16);
	mtype *m = (mtype *)ptr;
	unsigned int *g2 = (unsigned int *)(ptr + nbytes + 12);

	m->magic = MAGIC;
	m->sz = nbytes;

	// m->data should automatically point to the right place

	m->g1 = GUARD;
	*g2 = GUARD;

	return m->data;
	}

// arg1 is ptr to either mtype or normal buffer from old malloc()
void *
Realloc(void *udata, int newbytes)
	{
	mtype *m = GetMtypeFromPtr(udata);
	unsigned char *mptr = (unsigned char *)m;
	unsigned int *g2;

	if (! isMtype(m)) {
	   // Not a Mtype, revert to old realloc
	   printf("Realloc: missing signature, revert to old realloc()\n"); fflush(stdout);
	   return realloc(udata,newbytes);
	   }

	if (!CheckSignature(m)) {
	   printf("Realloc: invalid ptr\n");
	   fflush(stdout);
	   return NULL;
	   }

	if (! realloc(mptr, newbytes + 16)) {
	   printf("Realloc: Out of memory reallocing %d bytes (current size %d)\n",newbytes,m->sz);
	   fflush(stdout);
	   return NULL;
	   }

	m->sz = newbytes;
 	g2 = (unsigned int *)(mptr + newbytes + 12);
	*g2 = MAGIC;

	return (void *)m->data;
	}
	
void *
ZeroMalloc(int nbytes)
	{
	char *buf = (char *)Malloc(nbytes);
	bzero((void *)buf,nbytes);
	return (void *)buf;
	}

// valid cobinations are:
//
//	char *  	char *
//	char *		mtype *
//	mtype *		char *
//	mtype *		mtype *
char *
Strcpy(char *dst, char *src)
	{
	mtype *ms = GetMtypeFromPtr(src);
	mtype *md = GetMtypeFromPtr(dst);

	if (!isMtype(ms)) {
	   if (! isMtype(md)) return strcpy(dst,src);
	   else {
		if (strlen(dst) + strlen(src) >= md->sz) dst = Realloc(dst,md->sz + strlen(src) + 32);
		return strcpy(dst,src);
		}
	   }
	else {
	   if (! isMtype(md)) return strcat(dst,src);
	   else {
		if (strlen(dst) + strlen(src) >= md->sz) dst = Realloc(dst,md->sz + strlen(src) + 32);
		return strcpy(dst,src);
		}
	   }

	// Notreached
	printf("Strcpy: reached end of function\n"); fflush(stdout);
	exit(1);
	return NULL;
	}

char *
Strcat(char *dst, char *src)
	{
	mtype *ms = GetMtypeFromPtr(src);
	mtype *md = GetMtypeFromPtr(dst);

	if (!ms) {
	   printf("Strcat: invalid src ptr\n");
	   fflush(stdout);
	   return NULL;
	   }

	if (!md) {
	   printf("Strcat: invalid dst ptr\n");
	   fflush(stdout);
	   return NULL;
	   }

	// Check that the combined string fits in the allocated space
	if (strlen(src)+strlen(dst) >= md->sz) {
	   printf("Strcat: catenated string overflows allocated space %d bytes\n",md->sz);
	   return NULL;
	   }

	return strcat(dst,src);
	}

char *
Strdup(char *src)
	{
	mtype *m = GetMtypeFromPtr(src);
	unsigned char *dst;

	if (!m) {
	   printf("Strdup: invalid ptr\n");
	   fflush(stdout);
	   return NULL;
	   }

	// Check that the string fits in the allocated space
	if (strlen(src) >= m->sz) {
	   printf("Strdup: src string '%s' overflows allocated space %d bytes\n",src,m->sz);
	   return NULL;
	   }

	dst = Malloc(m->sz);
	strcpy(dst,src);
	return dst;
	}
	
// Given a normal string, return a String (MType object)
char *
String(char *str)
	{
	char *ptr = Malloc(strlen(str) + 8);
	Strcpy(ptr,str);
	return  ptr;
	}
	
int
Free(void *ptr)
	{
	mtype *m = GetMtypeFromPtr(ptr);

	if (! m) {
	   printf("Free: invalid ptr\n");
	   fflush(stdout);
	   return 0;
	   }

	free((unsigned char *)m);
	return 1;
	}

#endif
