#include "ninox.h"

/*
 * Read a P6 PPM file on standard input, and write a P6 PPM file on standard output.
 * The output file will have the object centered, with the assumption that any
 * pixels with values >= 10% R/G/B are part of the object, and elsewise are background
 */

#if UINT32_MAX == UINTPTR_MAX
#define STACK_CHK_GUARD 0xe2dee396
#else
#define STACK_CHK_GUARD 0x595e9fbd94fda766
#endif

uintptr_t __stack_chk_guard = STACK_CHK_GUARD;

__attribute__((noreturn))
void __stack_chk_fail(void)
{
#if __STDC_HOSTED__
        abort();
#elif __is_myos_kernel
        panic("Stack smashing detected");
#endif
}


int FilesProcessed = 0;

static int ProcessStdin(void);
static char * NextInSequence(char *last, char *nsfx);
static int CompareSer(const void *ptr1, const void *ptr2);

main(int argc, char **argv)
	{
	FILE *z;
	char fname[1024],*ptr;
	int i,j;
	unsigned int Start,Run;
	extern int ShowHist_PrintToStdout;
	extern int SER_QCount[];
	extern char *SER_Filename_map[];
	static char LastArg[1024];
	extern char *BINARYNAME;
	extern char *SyncBinary;
	extern int CurrentFileNumber;
	
	BINARYNAME = "ninox";
	OutDir = Malloc(1024);
	Strcpy(OutDir,".");

	ShowHist_PrintToStdout = 1;
	LastArg[0] = 0;

	BlankImageCount=0;
	fname[0]=0;
	Start = time(0);
	Srandom();

	printf("%s\n",argv[0]); fflush(stdout);

	// Look for a binary named "sync.exe" in the same dir as ninox.exe
	SyncBinary = NULL;
	ptr = argv[0];
	i = strlen(ptr);
	while(i && ptr[i-1]!='/' && ptr[i-1]!='\\') {
	   if (! strncasecmp(ptr+i-1,"ninox.exe",9)) {
		SyncBinary = Strdup(argv[0]);
		strcpy(SyncBinary+i-1,"sync.exe");
		if (isFile(SyncBinary)) {
		   Print("Sync found: %s\n",SyncBinary); fflush(stdout);
		   break;
		   }
		Free(SyncBinary);
		SyncBinary = NULL;
		}
	   --i;
	   }
	   

	// If called with no arguments then print all possible switches and exit
	if (argc==1) {
	   _CMD_help();
	   exit(0);
	   }

	// Init channel related global arrays
	for(i=0; i<NUM_CHANNELS; ++i) {
	   SER_QCount[i] = 0;
	   SER_Filename_map[i] = NULL;
	   }

	// Init the stacking module
	stack_init();

	if (StreamFilter) init_streamfilter();

	for(i=1; i < argc; ++i) {
	   char *arg = argv[i];

	   if (arg[0]=='-') {
		// set VAR=VALUE
		int n = SetVariable(arg+1);
		if (n == 0 ) {
		   Print("%s: Unknown switch\n",arg);
		   exit(1);
		   }
		if (n < 0) {
		   Print("Error in switch %s\n",arg);
		   exit(1);
		   }
		continue;
		}

	   if (arg[0] == '+' && LastArg) {
	 	char *next = NextInSequence(LastArg,arg+1);
		if (next) {
		   Print("Found next file next=[%s]\n",next);
		   arg = next;
		   Print("Found next file arg=[%s]\n",arg);
		   }
		else {
		   Print("Could not find next in sequence for [%s] [%s]\n",LastArg,arg+1);
		   exit(1);
		   }
		}

	   // Remove trailing '/' or '\'
	   while(strlen(arg)>0 && arg[strlen(arg)-1] == '/' || arg[strlen(arg)-1] == '\\') 
		   arg[strlen(arg)-1] = 0;

	   if (isFile(arg) && ImageCount > 0) {
		int filecount=0;

		// If this is an ini file then load it
		if (!strcasecmp(arg+strlen(arg)-4,".ini")) {
		   Print("Loading %s\n",arg);
		   LoadConfigFile(arg,"ninox",NULL,NULL);
		   continue;
		   }

		// Check if there's a .txt firecapture logfile and load it
		//Print("Loading from [%s]\n",arg); fflush(stdout);
		LoadLogfile(arg,NULL);

		//Print("ok\n"); fflush(stdout);

		if (! LoadedConfig) {
		   // no explicit ninox.ini given, so try to find one based on the Profile from
		   // the .txt logfile
		   char *ini = ZeroMalloc(1024);
		   char *pro = GetLogEntry("Profile");

		   if (pro) {
		     // Linux home dir
		     char *home = getenv("HOME");
		     if (home && isDirectory(home)) {
			strcpy(ini,home);
			strcat(ini,"/.ninox/");
			strcat(ini,pro);
			strcat(ini,".ini");
			}

		    if (! isFile(ini)) {
			// MS Windows home dir
			*ini=0;
			char *hd = getenv("HOMEDRIVE");
			char *hp = getenv("HOMEPATH");
			if (hd && hp) {
			   strcpy(ini,hd);
			   if (strlen(hp)>1) strcat(ini,hp);
			   else {
				strcat(ini,"/Users/");
			   	strcat(ini,getenv("LOGNAME"));
				}
			   strcat(ini,"/ninox/");
			   strcat(ini,pro);
			   strcat(ini,".ini");
			   }
			}

		   if (isFile(ini)) {
			Print("Loading ninox config file [%s]\n",ini);
			LoadConfigFile(ini,"ninox",NULL,NULL);
			}
		   }

		// Last resort - look for a ninox.ini in the current directory
		if (isFile("./ninox.ini")) {
		   Print("Loading ninox config file ./ninox.ini\n");
		   LoadConfigFile("ninox.ini","ninox",NULL,NULL);
		   }
		}
			
		// Remember the current filename
		Strcpy(LastArg,arg);

		/* Must be a file to align */
		if (! DoLoadQFiles) {
	 	   filecount = Align(arg);
	 	   if (! filecount) {
		      Print("Alignment failed on [%s]\n",arg);
		      exit(1);
		      }
		   --ImageCount;
		   FilesProcessed += filecount;
		   CurrentFileNumber++;
		   }
		}
	   else if (isDirectory(arg)) {
		if (! DoLoadQFiles) FilesProcessed += ProcessDir(arg);
		}
	   else {
		Print("Unknown argument: '%s'\n",arg);
		exit(1);
		}
	   }
	   
	if (InteractiveMode) {
	   Interact();
	   exit(0);
	   }

	if (DoLoadQFiles) {
		LoadQFiles();
		Run = time(0) - Start;
		printf("running time = %02dm%02ds seconds\n",Run/60,Run%60);
		fflush(stdout);
		exit(0);
		}

	if (FilesProcessed == 0 && UsingStdin) 
		FilesProcessed += ProcessStdin();

	// No files specified? Then process the current directory by default
	if (FilesProcessed == 0 && ! UsingStdin)
	   FilesProcessed += ProcessDir(".");

	if (! FilesProcessed) 	Print("Found no files to process.\n");
	else 			Print("\n  -> Processed %d files\n",FilesProcessed);

	fflush(stdout);

        if (StackCount>0 && StackFile != NULL) {
	  if (OutDir && !strstr(StackFile,"\\") && !strstr(StackFile,"/"))
		sprintf(fname,"%s/%s",OutDir,StackFile);
	  else	Strcpy(fname,StackFile);
	  if (OutDir) Mkdir(OutDir);
          if (!Quiet) {Print("  -> Writing stackfile %s from %d frames\n",fname,StackCount); fflush(stdout); }

          write_stack_data(fname);
          }

	if (QEstimator) {
	   int do_trim = 0;
	   char qfname[32];

           if (ProcessOffset || ProcessSkip)
                sprintf(qfname,"quality-%d-%d.txt",ProcessOffset,ProcessSkip);
           else  Strcpy(qfname,"quality.txt");

           WriteQFile(qfname);

	   if (ProcessOffset || ProcessSkip) {
		Print("child process detected, renumbering skipped\n");
		}
	   else {
	      if (QTrim_Before || QTrim_After) do_trim = 1;

	      if (do_trim) Mark_Trim(QTrim_Before,QTrim_After);

	      if (QRenumberFiles) {
		Print("\nRenumbering...\n"); fflush(stdout);
		QRenumber(QRenumberFiles);
		}
	      else if (do_trim) Delete_Trim();
	      }
	   }

	// If we were writing a global ser file then make sure it gets closed
	if (WriteSer && WriteSerHandle) {
	   Print("Closing output ser file '%s'\n",WriteSerFile);
	   CloseSerFile(WriteSerHandle);
	   }

	// Save the commandline
	z = OpenLogfile("ninox.run","ab");
	if (z != NULL) {
	   for(i=0; i<argc; ++i)
	      if (argv[i][0] == '-' || argv[i][0]=='+') fprintf(z,"%s ",argv[i]);
	   fprintf(z,"\n");
	   fclose(z);
	   }
	else {
	   Print("Cannot save settings file ninox.run\n",fname); fflush(stdout);
	   }

	if (DetectHotPixels) {
	   WriteHotPixelFile("hotpixels.txt");
	   }

	Run = time(0) - Start;
	printf("running time = %02dm%02ds seconds\n",Run/60,Run%60);
	fflush(stdout);
	exit(0);
	}

// Assume there is no path component on these filenames
static int
CompareSer(const void *ptr1, const void *ptr2)
	{
	char **p1 = (char **)ptr1;
	char **p2 = (char **)ptr2;
	int have1=0;
	int have2=0;
	int year1,year2,month1,month2,day1,day2;
	int hr1,hr2,min1,min2,sec1,sec2;

	if (!have1 && sscanf(*p1,"%04d%02d%02d_%02d%02d%02d_",&year1,&month1,&day1,&hr1,&min1,&sec1) == 6) have1 = 1;
	if (!have1 && sscanf(*p1,"%04d-%02d-%02d-%02d%02d%02d_",&year1,&month1,&day1,&hr1,&min1,&sec1) == 6) have1 = 1;
	if (!have1 && sscanf(*p1,"%04d-%02d-%02d-%02d%02d_%d",&year1,&month1,&day1,&hr1,&min1,&sec1) == 6) { have1 = 1; sec1 *= 6; }

	if (!have2 && sscanf(*p2,"%04d%02d%02d_%02d%02d%02d_",&year2,&month2,&day2,&hr2,&min2,&sec2) == 6) have2 = 1;
	if (!have2 && sscanf(*p2,"%04d-%02d-%02d-%02d%02d%02d_",&year2,&month2,&day2,&hr2,&min2,&sec2) == 6) have2 = 1;
	if (!have2 && sscanf(*p2,"%04d-%02d-%02d-%02d%02d_%d",&year2,&month2,&day2,&hr2,&min2,&sec2) == 6) { have2 = 1; sec2 *= 6; }

	if (have1 && have2) {
	   if (year1 != year2) return year1 - year2;
	   if (month1 != month2) return month1-month2;
	   if (day1 != day2) return day1-day2;
	   if (hr1 != hr2) return hr1-hr2;
	   if (min1 != min2) return min1-min2;
	   if (sec1 != sec2) return sec1-sec2;
	   }
	return 0;
	}

// Take the current file and a new suffix, 
// look for the next one in chronological order
static char *
NextInSequence(char *last, char *nsfx)
	{
	char *ftype = last + strlen(last) - 4;
	char *dir = Strdup(last);
	char *last_fname = NULL;
	char *ptr,*rval=NULL;
	char *entries[1024];
	struct dirent *e;
	int i,j,ec=0;
	DIR *d;

	// prune off the filename to get the dir
	ptr = dir + strlen(dir) - 1;
	while(ptr != dir && *(ptr) != '/' && *(ptr) != '\\') --ptr;
	if (ptr != dir) {
	   *ptr=0;
	   last_fname = ptr+1;
	   }
	else {
	   // No directory given, assume we're in the directory
	   last_fname = dir;
	   dir = ".";
	   }

	if (! isDirectory(dir)) {
	   Print("NextInSequence: Error: [%s] not a directory, from %s\n",dir,last);
	   return NULL;
	   }

	// Open and read all dir entries, keep those with the same suffix
	// as last
	d = opendir(dir);
	while(e = readdir(d)) if (strlen(e->d_name)>4) {
	   char *eftype = e->d_name + strlen(e->d_name) - 4;
	   if (!strcasecmp(eftype,ftype)) {
		entries[ec] = Malloc(1024);
		Strcpy(entries[ec],e->d_name);
		//Print("Loading [%s]\n",entries[ec]); fflush(stdout);
		ec++;
		}
	   }
	closedir(d);

	// Sort the dir entries on datetime
	qsort(entries,ec,sizeof(char *),CompareSer);

	//Print("\n\n");
	//for(i=0; i<ec; ++i) Print("Sorted: [%s]\n",entries[i]); Print("\n\n"); fflush(stdout);

	// Run through the list to find our current entry and then look for the
	// next entry with the new suffix.
	for(i=0; i<ec && !rval; ++i) if (!strcasecmp(last_fname,entries[i])) {
	   for(j=i+1; j<ec && !rval; ++j) {
		char *ptr = entries[j]+strlen(entries[j])-4-strlen(nsfx);
		if (!strncasecmp(ptr,nsfx,strlen(nsfx))) {
		   rval = Malloc(strlen(dir)+strlen(entries[j]+2));
		   sprintf(rval,"%s/%s",dir, entries[j]);
		   }
		}
	   }

	for(i=0; i<ec; ++i) Free(entries[i]);
	return rval;
	}


static int
comp(const void *ptr1, const void *ptr2)
	{
	char **s1 = (char **)ptr1;
	char **s2 = (char **)ptr2;
	char *p1,*p2;

	char *e1 = *s1 + strlen(*s1);	// ends of the strings
	char *e2 = *s2 + strlen(*s2);

	int i1,i2;

	// seek to end of string
	p1 = e1;
	p2 = e2;

	// seek back until we find the last path entry
	while(p1 != *s1 && *(p1-1) != '/' && *(p1-1) != '\\') --p1;
	while(p2 != *s2 && *(p2-1) != '/' && *(p2-1) != '\\') --p2;
	
	if (strcasecmp(e1-4,".fta")==0 && strcasecmp(e2-4,".fta")==0) {
	   int a1,a2,b1,b2,c1,c2;

	   // first try is based on the naming convention for .fta files
	   if (sscanf(p1,"data%d-%d-%d.fta",&a1,&b1,&c1) == 3 &&
	       sscanf(p2,"data%d-%d-%d.fta",&a2,&b2,&c2) == 3) {
		if (a1 != a2) return a1-a2;
		if (b1 != b2) return b1-b2;
		return c1-c2;
		}

	   // second try is normal string-based
	   return strcasecmp(*s1,*s2);
	   }

	// Go forward until we find a digit
	while(*p1 && ! isdigit(*p1)) ++p1;
	while(*p2 && ! isdigit(*p2)) ++p2;

	// Hopefully we are now on a substring that can be used for comparison
	i1 = atoi(p1);
	i2 = atoi(p2);

	if (i1<i2) return -1;
	if (i1>i2) return 1;
	return 0;
	}

// Look in the named directory for files to process
int
ProcessDir(char *dir)
	{
	char fname[256];
	struct dirent *e;
	DIR *d = opendir(dir);
	int i,count=0,prepend_dir,FileCount;
	char *file_list[DIR_MAX_COUNT];
	char OutDir_Save[1024];
	extern int CurrentFileNumber;

	Print("ProcessDir %s, OutDir='%s'\n",dir,OutDir);

	if (d==NULL) {
	   Print("Error opening directory %s\n",dir);
	   return 0;
	   }

	// Munch off trailing slash
	i = strlen(dir) - 1; 
	if (dir[i] == '/' || dir[i] == '\\') dir[i] = 0;

	if (dir[0]=='.' && dir[1]==0)
	   prepend_dir = 0;
	else
	   prepend_dir = 1;

	// If we have -outdir-append-src then append the last component
	// of d onto OutDir
	OutDir_Save[0]=0;
	if (OutDir && OutDirAppendSrc) {
	   int i = strlen(dir)-1;

	   Print("OutDirAppend\n");
	   while(i>0 && dir[i-1] != '/' && dir[i-1] != '\\') --i;
	   if (i) {
	   	Strcpy(OutDir_Save,OutDir);
		Strcat(OutDir,"/");
		Strcat(OutDir,dir+i);
		printf("Using '%s' for OutDir\n",OutDir);
	   	}
	   }
	else if (OutDir && OutDirPrependSrc) {
	   // If we have -outdir-prepend-src then prepend the last component
	   // of d before OutDir
	   int i = strlen(dir)-1;

	   Print("Prepend\n"); fflush(stdout);
	   while(i>0 && dir[i-1] != '/' && dir[i-1] != '\\') --i;
	   if (i) {
		   Print("Changing OutDir\n");
	   	Strcpy(OutDir_Save,OutDir);
		Strcpy(OutDir,"./");
		if (! NoSave) Mkdir(dir+i);
		Strcat(OutDir,dir+i);
		Strcat(OutDir,"/");
		Strcat(OutDir,OutDir_Save);
		Print("Using '%s' for OutDir\n",OutDir);
	   	}
	   }
	
	if (OutDir && ! NoSave) Mkdir(OutDir);

	while(e = readdir(d)) {
	   if (strlen(e->d_name)>4) {
		if (prepend_dir) sprintf(fname,"%s/%s",dir,e->d_name);
		else Strcpy(fname,e->d_name);

		if (LoadedConfig==0 && strcmp(e->d_name,"ninox.ini")==0) {
		   Print("Loading default config file %s\n",fname);
		   LoadConfigFile(fname,"ninox",NULL,NULL);
		   }

	   	if (! access(fname,R_OK|F_OK) && 
		   (	strcasestr(fname,".bmp") ||
		   	strcasestr(fname,".ppm") || strcasestr(fname,".pgm") ||
			strcasestr(fname,".fit") || strstr(fname,".fta") ||
			strcasestr(fname,".ser"))) {
		   	file_list[count++] = Strdup(fname);
			if ((count%1000)==0) {
				Print("Loaded %d files...\n",count);
				fflush(stdout);
				}
			}

		if (count>=DIR_MAX_COUNT) {
		   Print("Max count of %d files reached!\n",DIR_MAX_COUNT);
		   exit(1);
		   }
		}
	   }
	closedir(d);

	// Turn off archive chaining since we've explicitly found
	// all the files.
	ChainArchives = 0;
	qsort(file_list,count,sizeof(char *),comp);

	for(i=0; i<count; ++i) Print("Added %s\n",file_list[i]);

	for(i=FileCount=0; i<count; ++i) {
	   char *fname = file_list[i];
	   int fc = 0;

	   if (ImageCount > 0) {
	      fc = Align(fname);
	      if (! fc) Print("Alignment failed on [%s]\n",fname);
	      FileCount += fc;
	      CurrentFileNumber++;
	      if (ImageCount < IMAGECOUNT_MAX) {
	         Print("%d images remaining\n",ImageCount);
		 }
	      }

	   Free(file_list[i]);
	   }

	if (OutDir && OutDir_Save) {
	   Strcpy(OutDir, OutDir_Save);
	   }

	return FileCount;
	}

// We're reading a stream of frames from standard input
static int
ProcessStdin()
	{
	int i,fc;
	int FileCount=0;
	extern int CurrentFileNumber;

	for(i=FileCount=0; i<IMAGECOUNT_MAX; ++i) {
	   if (ImageCount > 0) {
	      fc = Align("-");
	      if (fc <= 0) {
		  Print("Alignment failed on STDIN\n");
		  return FileCount;
	   	  }

	      FileCount++;
	      ImageCount--;
	      if (ImageCount < IMAGECOUNT_MAX) {
	         Print("%d images remaining\n",ImageCount);
		 }
	      }

	   }

	CurrentFileNumber++;
	return FileCount;
	}
