#include "astrolib.h"
#include <string.h>

int ifmt = FMT_BMP;
int FrameTotal = 0;
int BYTES_PER_PIXEL = 3;

uchar Zero[4] = {0,0,0,0};

inline double PixDiffLC(uchar *p1, uchar *p2, int depth, double l, double c);

int WriteBMPHeader24(FILE *out,int width, int height) { return WriteBMPHeader(out,width,height,24,NULL,0); }

int WriteBMPHeader(FILE *out,int width, int height, int depth, unsigned char *map, int entries)
	{
	bmp_header H;
	int pad = (4 - (width & 3)) & 3;
	int bufsize = (width+pad) * height * 3;

        if (! fwrite("BM",2,1,out)) {
                fprintf(stderr,"Short write\n");
                exit(1);
                }

	if (depth == 8) {
	   H.filesz = 54 + 1024 + bufsize;
	   H.offset = 54 + 1024;
	   H.bitcount = 8;
	   }
	else if (depth == 24) {
           H.filesz = 54 + bufsize;
           H.offset = 54;
           H.bitcount = 24;
	   }
	else {
	   fprintf(stderr,"Unsupported depth: %d in WriteBMPHeader\n",depth);
	   exit(1);
	   }

        H.r1 = H.r2 = 0;
        H.size = 40;
        H.width = width;
        H.height = height;
        H.planes = 1;
        H.compression = 0;
        H.sizeimage = 0;
        H.xppm = 0;
        H.yppm = 0;
        H.cmapentries = entries;
        H.clrimportant = 0;

        if (! fwrite((char *)&H,sizeof(H),1,out)) {
                fprintf(stderr,"cannot write to output\n");
                exit(1);
                }

	if (depth==8) {
	   if (! fwrite(map,1024,1,out)) {
	 	fprintf(stderr,"Short write on colourmap in WriteBMPHeader\n");
		exit(1);
		}
	   }

	return(0);
        }

int
WritePGMHeader8(FILE *out, int width, int height)
	{
	fprintf(out,"P5\n%d %d\n255\n",width,height);
	return(0);
	}

int
WritePGMHeader16(FILE *out, int width, int height)
	{
	fprintf(out,"P5\n%d %d\n65535\n",width,height);
	return(0);
	}

int
WritePPMHeader(FILE *out, int width, int height)
	{
	fprintf(out,"P6\n%d %d\n255\n",width,height);
	return(0);
	}

int ReadPPMHeader(FILE *in,char *type, int *width, int *height, int *colours)
	{
	int ch,i,incomment;
	char buf[32];

	type[0]=0; *width=0; *height=0; *colours=0;
	incomment=0; i=0;
	while(1)
	   {
	   switch(ch=getc(in))
		{
		case EOF: return 1;
		case '#': incomment=1; break;
		case ' ': case '\t':
			if (incomment) break;
		case '\n': case '\r':
			if (incomment) {incomment=0; break;}
			if (type[0]==0)
			   {
			   strcpy(type,buf);
			   if (type[0]!='P') return 1;
			   if (type[1]!='3' && type[1]!='6') return 1;
			   i=0; buf[0]=0; break;
			   }
			if (*width==0) {*width=atoi(buf); i=0; buf[0]=0; break;}
			if (*height==0) {*height=atoi(buf); i=0; buf[0]=0; break;}
			if (*colours==0) {*colours=atoi(buf); return(0); break;}
		default:
			buf[i++]=ch; buf[i]=0; break;
	   	}
	   }
	return 1;
	}

int
ReadBMPHeader(FILE *in, char *type, int *width, int *height, int *depth, 
	unsigned char *map, int *entries)
	{
	bmp_header H;
	int i;

	*type = 0;
	*width = 0;
	*height = 0;
	*entries = 0;

	if (sizeof(H) != 52) {
	   fprintf(stderr,"Struct bmp_header not 52 bytes\n");
	   exit(1);
	   }

	/* Look for magic */
	if (! fread(type,2,1,in)) {
	   fprintf(stderr,"Short read\n");
	   exit(1);
	   }

	type[2] = 0;
	if (strncmp(type,"BM",2)) {
	   fprintf(stderr,"Missing BM header - not a BMP file?\n");
	   exit(1);
	   }

	/* Read the rest of the header */
	if (! fread((char *)&H,sizeof(H),1,in)) {
	   fprintf(stderr,"Short read\n");
	   exit(1);
	   }

	/* Header remaining size shoulf be 40 bytes */
	if (H.size != 40) {
	   fprintf(stderr,"Header size %d not 40 bytes\n",H.size);
	   exit(1);
	   }

	*depth = H.bitcount;
	*entries = H.cmapentries;

	if (*depth == 24) {
	   /* Offset to image data should be 54 */
	   if (H.offset != 54) {
	      fprintf(stderr,"Image data offset %d not 54 bytes\n",H.offset);
	      exit(1);
	      }

	   /* sanity check */
	   if (H.planes == 1 && H.bitcount == 24) {
	      *width = H.width;
	      *height = H.height;
	      return 1;
	      }
	   else {
	      	fprintf(stderr,"Error in header: planes=%d, bitcount=%d\n",H.planes,H.bitcount);
		exit(1);
	      	}
	   }

	if (*depth == 8) {
	   *entries = 256;

	   /* Offset to image data should be 54  + colourmap(1024)*/
	   if (H.offset != 1024+54) {
	      fprintf(stderr,"Image data offset %d not %d bytes\n",H.offset,1024+54);
	      exit(1);
	      }

	   /* Read colourmap */
	   if (! fread(map, (*entries) * 4,1, in)) {
		fprintf(stderr,"Short read on colour map\n");
		exit(1);
	 	}

	   *width = H.width;
	   *height = H.height;

	   return 1;
	   }

	return 0;
	}

// Load an image file and return as a 24bpp image
unsigned char *
LoadImage24(FILE *in, int *width, int *height)
	{
	unsigned char cmap[1024];
	int i,depth,cmapentries;
	unsigned char *buffer;
	unsigned char *buf24,*ptr;

	buffer = LoadImage(in,width,height,&depth,cmap,&cmapentries);
	if (depth==24) return buffer;
	else if (depth==8 && cmapentries) {
	   int npix = *width * *height;

	   buf24 = malloc(npix * 3);
	   if (buf24 == NULL) {
		fprintf(stderr,"Out of memory in LoadImage24\n");
	 	exit(1);
		}

	   ptr = buf24;
	   for(i=0; i < npix; ++i) {
		unsigned char *o = cmap + buffer[i] * 4;
		*(ptr++) = *(o++);
		*(ptr++) = *(o++);
		*(ptr++) = *(o++);
		}

	   free(buffer);
	   return buf24;
	   }
	else {
	   fprintf(stderr,"Unsupported format: %dx%dx%d\n",*width,*height,depth);
	   exit(1);
	   }

	// notreached
	return (unsigned char *) NULL;
	}

unsigned char *
LoadImage(FILE *in, int *width, int *height, int *depth, unsigned char *colourmap, int *cmapentries)
	{
	int colourmax;
	char type[10];
	unsigned char *buffer = NULL;
	int bufsize,err;

	/*
	 * Read the input
	 */

	type[0] = 0;
	*cmapentries = 0;

	if (ifmt == FMT_PPM) {
	   ReadPPMHeader(in,type,width,height,&colourmax);

	   if (strcmp(type,"P6")) {
	      fprintf(stderr,"Type '%s' not understood\n",type);
	      Usage();
	      exit(1);
	      }
	   *depth = 24;
	   *cmapentries = 0;
	   }
	else if (ifmt == FMT_BMP) {
	   ReadBMPHeader(in,type,width,height,depth,colourmap,cmapentries);

	   if (strcmp(type,"BM")) {
		fprintf(stderr,"Input is not a BMP\n");
		exit(1);
		}
	   }

	if (*width > WIDTH || *height > HEIGHT) {
	   fprintf(stderr,"Only support %dx%d at the moment\n",WIDTH,HEIGHT);
	   exit(0);
	   }

	/*
	 * Allocate the buffer and load the data
	 */
	if (*depth == 24) {
	   bufsize = (*width) * (*height) * 3;
	   }
	else if (*depth == 8) {
	   bufsize = (*width) * (*height);
	   }
	   
	buffer = (unsigned char *)malloc(bufsize);
	if (buffer == NULL) {
	   fprintf(stderr,"Cannot malloc buffer of %d bytes\n",bufsize);
	   exit(1);
	   }

	err = fread(buffer,bufsize,1,in);

	if (err != 1) {
	   fprintf(stderr,"Error in reading data, only read %d bytes of %d\n",err,bufsize);
	   exit(1);
	   }

	FrameTotal++;
	return buffer;
	}

char *
BaseName(char *name)
	{
	char *b = name;

	/* Find the base filename component */
	while(*b) b++;
	while(*b != '/' && *b != '\\' && b != name) --b;
	if (*b == '/' || *b == '\\') ++b;

	return b;
	}

inline double
PixDiffLC24(uchar *p1, uchar *p2, double l, double c)
	{
	return PixDiffLC(p1,p2,24,l,c);
	}

/*
 * Distance in colourspace between the pixels, normalised to range (0.0) - (1.0)
 */
inline double
PixDiffLC(uchar *p1, uchar *p2, int depth, double l, double c)
	{
	int b,g,r;

	if (depth==24) {
	   b = *(p1++) - *(p2++);	// 0.0 - 255.0
	   g = *(p1++) - *(p2++);	// 0.0 - 255.0
	   r = *(p1++) - *(p2++);	// 0.0 - 255.0
	   double L,C;

	   r *= r; g *= g; b *= b;

	   // Luminance difference
	   L = r * 0.299 + g * 0.587 + b * 0.114;
	   L *= L;

	   // Colourspace difference
	   C = r + g + b;
	   C *= C;

	   // Weighted result, normalised (0.0 - 1.0)
	   return (L * l + C * c) / (195075.0 * (l+c));
	   }

	if (depth==8) {
	   b = *(p1++) - *(p2++);
	   return ((double)b * (double)b) / 65025.0;
	   }
	}

int
FFT2D_bmp(unsigned char *bmp, double *buffer,int width, int height)
	{
	int n = width * height;
	int i;
	COMPLEX *fftbuf;
	COMPLEX *fftptr;

	/* check that width and height are powers of 2 */
	if (width & (width-1)) {
	   fprintf(stderr,"Input width is not a power of 2\n");
	   return(0);
	   }

	if (height & (height-1)) {
	   fprintf(stderr,"Input height is not a power of 2\n");
	   return(0);
	   }

	fftbuf = malloc(sizeof(COMPLEX) * n);
	if (fftbuf == NULL) {
	   fprintf(stderr,"Not enough memory for FFT buffer\n");
	   return 0;
	   }

	/* Convert the BGR data into a complex input buffer */
	fftptr = fftbuf;
	for(i=0; i<n; ++i,++fftptr) {
	   fftptr->real = *(bmp++) * B_LUM + *(bmp++) * G_LUM + *(bmp++) * R_LUM;
	   fftptr->imag = 0;
	   }

	/* Inplace FFT */
	if (! FFT2D(fftbuf, width, height, 1)) {
	   free(fftbuf);
	   return(FALSE);
	   }

	/* Copy data into output buffer */
	fftptr = fftbuf;
	for(i=0; i<n; ++i,++fftptr) {
	   buffer[i] = fftptr->real;
	   }

	free(fftbuf);
	return(1);
	}

/*-------------------------------------------------------------------------
   Perform a 2D FFT inplace given a complex 2D array
   The direction dir, 1 for forward, -1 for reverse
   The size of the array (nx,ny)
   Return false if there are memory problems or
      the dimensions are not powers of 2
*/
int FFT2D(COMPLEX *c,int nx,int ny,int dir)
{
   int i,j,o;
   int m,twopm;
   double *real,*imag;

   /* Transform the rows */
   real = (double *)malloc(nx * sizeof(double));
   imag = (double *)malloc(nx * sizeof(double));

   if (real == NULL || imag == NULL) return(FALSE);
   if (!Powerof2(nx,&m,&twopm) || twopm != nx) return(FALSE);

   for (o=j=0;j<ny;j++,o+=nx) {
      for (i=0;i<nx;i++) {
         real[i] = c[o+i].real;
         imag[i] = c[o+i].imag;
      }
      FFT(dir,m,real,imag);
      for (i=0;i<nx;i++) {
         c[o+i].real = real[i];
         c[o+i].imag = imag[i];
      }
   }
   free(real);
   free(imag);

   /* Transform the columns */
   real = (double *)malloc(ny * sizeof(double));
   imag = (double *)malloc(ny * sizeof(double));
   if (real == NULL || imag == NULL) return(FALSE);
   if (!Powerof2(ny,&m,&twopm) || twopm != ny) return(FALSE);
   for (i=0;i<nx;i++) {
      for (o=i,j=0;j<ny;j++,o+=nx) {
         real[j] = c[o].real;
         imag[j] = c[o].imag;
      }
      FFT(dir,m,real,imag);
      for (o=i,j=0;j<ny;j++,o+=nx) {
         c[o].real = real[j];
         c[o].imag = imag[j];
      }
   }
   free(real);
   free(imag);

   return(TRUE);
}

/*-------------------------------------------------------------------------
   This computes an in-place complex-to-complex FFT
   x and y are the real and imaginary arrays of 2^m points.
   dir =  1 gives forward transform
   dir = -1 gives reverse transform

     Formula: forward
                  N-1
                  ---
              1   \          - j k 2 pi n / N
      X(n) = ---   >   x(k) e                    = forward transform
              N   /                                n=0..N-1
                  ---
                  k=0

      Formula: reverse
                  N-1
                  ---
                  \          j k 2 pi n / N
      X(n) =       >   x(k) e                    = forward transform
                  /                                n=0..N-1
                  ---
                  k=0
*/
int FFT(int dir,int m,double *x,double *y)
{
   long nn,i,i1,j,k,i2,l,l1,l2;
   double c1,c2,tx,ty,t1,t2,u1,u2,z;

   /* Calculate the number of points */
   nn = 1;
   for (i=0;i<m;i++)
      nn *= 2;

   /* Do the bit reversal */
   i2 = nn >> 1;
   j = 0;
   for (i=0;i<nn-1;i++) {
      if (i < j) {
         tx = x[i];
         ty = y[i];
         x[i] = x[j];
         y[i] = y[j];
         x[j] = tx;
         y[j] = ty;
      }
      k = i2;
      while (k <= j) {
         j -= k;
         k >>= 1;
      }
      j += k;
   }

   /* Compute the FFT */
   c1 = -1.0;
   c2 = 0.0;
   l2 = 1;
   for (l=0;l<m;l++) {
      l1 = l2;
      l2 <<= 1;
      u1 = 1.0;
      u2 = 0.0;
      for (j=0;j<l1;j++) {
         for (i=j;i<nn;i+=l2) {
            i1 = i + l1;
            t1 = u1 * x[i1] - u2 * y[i1];
            t2 = u1 * y[i1] + u2 * x[i1];
            x[i1] = x[i] - t1;
            y[i1] = y[i] - t2;
            x[i] += t1;
            y[i] += t2;
         }
         z =  u1 * c1 - u2 * c2;
         u2 = u1 * c2 + u2 * c1;
         u1 = z;
      }
      c2 = sqrt((1.0 - c1) / 2.0);
      if (dir == 1)
         c2 = -c2;
      c1 = sqrt((1.0 + c1) / 2.0);
   }

   /* Scaling for forward transform */
   if (dir == 1) {
      for (i=0;i<nn;i++) {
         x[i] /= (double)nn;
         y[i] /= (double)nn;
      }
   }

   return(TRUE);
}

/*
 * return TRUE if i is a power of 2.
 * Set m to the power and tpm to (2 ^ m)
 */
int
Powerof2(int i, int *m, int *tpm)
	{
	if (i<2) return FALSE;
	if (i & (i-1)) return FALSE;

	*m=0; *tpm=1;
	while(i>1) {
	   i >>= 1;
	   ++(*m);
	   *tpm <<= 1;
	   }

	return TRUE;
	}

/*
 * Generate a SIMPLE FITS header
 */
int
Write2DFITSHeader(FILE *out, int width, int height, int bpp)
	{
	int line = 0;

	WriteFITSBoolean(out,"SIMPLE",TRUE);	++line;
	WriteFITSInteger(out,"BITPIX",16);	++line;
	WriteFITSInteger(out,"NAXIS",2);	++line;
	WriteFITSInteger(out,"NAXIS1",width);	++line;
	WriteFITSInteger(out,"NAXIS2",height);	++line;
	WriteFITSInteger(out,"BSCALE",1);	++line;
	WriteFITSInteger(out,"BZERO",0);	++line;
	WriteFITSEnd(out);			++line;

	/* We need 36 lines in the header */
	while(line++ < 36) WriteFITSBlankLine(out);
	}

/*
 * Booleans have KEYWORD in cols 1-8, col9 == '=' col10 == ' '
 * col30 is either 'T' or 'F'
 */

int
WriteFITSBoolean(FILE *out, char *keyword, int bool)
	{
	char bstr;

	if (bool) bstr = 'T'; else bstr = 'F';

	/* Generate an 80 character output unit */
	fprintf(out,"%-8.8s= %20s%50s",keyword,bstr,"");
	return(1);
	}

int
WriteFITSString(FILE *out, char *keyword, char *str)
	{
	int i;
	char buffer[81];

	buffer[0] = 0;

	if (strlen(str) > 65) {
	   fprintf(stderr,"FITSString: string too long: '%s'\n",str);
	   return(0);
	   }
	sprintf(buffer,"%-8.8s= '%s'",keyword,str);

	/* pad it out to 80 chars */
	for(i=strlen(buffer); i<80; ++i) buffer[i]=' ';
	buffer[i]=0;

	fprintf(out,"%s",buffer);
	return(1);
	}

int
WriteFITSInteger(FILE *out, char *keyword, int val)
	{
	char buffer[81];
	int i;

	buffer[0] = 0;
	sprintf(buffer,"%-8.8s= %20d",keyword,val);

	/* pad it out to 80 chars */
	for(i=strlen(buffer); i<80; ++i) buffer[i]=' ';
	buffer[i]=0;

	fprintf(out,"%s",buffer);
	return(1);
	}

int
WriteFITSEnd(FILE *out)
	{
	fprintf(out,"%-80s","END");
	return(1);
	}

int
WriteFITSBlankLine(FILE *out)
	{
	fprintf(out,"%80s","");
	return(1);
	}
