#include "ppmcentre.h"
#include <ctype.h>
#include <string.h>

/*
 * Generate a quality estimate for the given region by computing gradients
 * on downsampled images
 */

#define MAX_FILES 100000

typedef struct
	{
	int bucket;  // 0..6
	char *fname;
	char *newname;
	double val;
	} Sharp;

static int SubSample(unsigned char *buf, int img_wid, int bpp, int size);
static double AnalyseImage16(unsigned short *buf, int x_samples,int y_samples);

static double PixDiff(unsigned char *p1, unsigned char *p2);
inline static double PixDiffLC1(unsigned char *p1, unsigned char *p2, int depth, double l, double c);
static int Compare(const void *, const void *);

double cutoff = 0.5;
Sharp flist[MAX_FILES];
static int flist_idx = 0;

static int
Compare(const void * a, const void * b)
	{
	Sharp *A = (Sharp *)a;
	Sharp *B = (Sharp *)b;

	if (B->bucket != A->bucket)
	   return (B->bucket>A->bucket)?1:-1;

	if (B->val > A->val) return (1);
	if (B->val < A->val) return (-1);
	return 0;
	}
	
void
WriteQFile(char *fname)
	{
	FILE *out = fopen(fname,"w");
	int i;

	if (out == NULL) {
	   Print("WriteQFile: cannot open '%s' for writing\n",fname);
	   return;
	   }

	// Sort the list
	qsort(flist,flist_idx,sizeof(Sharp),Compare);

	for(i=0; i<flist_idx; ++i) {
	   fprintf(out,"%s\t%lf\n",flist[i].fname,flist[i].val);
	   }
	fclose(out);
	}

// Take the sorted list of files and renumber them
// arg is either QRENUMBER_FIRST or QRENUMBER_LAST, selecting which
// numeric portion of teh filename to change
void
QRenumber(int which)
	{
	int i,j,ndigits;
	int count=0;
	char *ptr=NULL,*newname=NULL,*p2=NULL;

	if (!Quiet) Print("\tCalculating new names...\n");
	for(i=0; i<flist_idx; ++i) {

	   // locate the start of the filename portion
	   ptr = flist[i].fname + strlen(flist[i].fname) - 1;
	   while(ptr != flist[i].fname && *(ptr-1) != '/') --ptr;

	   // ptr points to the start of the section that is to be replaced
	   if (which == QRENUMBER_FIRST) {
	      // locate first numeric section in the filename
	      while(*(ptr) && !isdigit(*ptr)) ++ptr;
	      }
	   else if (which == QRENUMBER_LAST) {
	      ptr = flist[i].fname + strlen(flist[i].fname) - 1;
	      while(ptr != flist[i].fname && *(ptr-1) != '/' && !isdigit(*ptr)) --ptr;
	      while (ptr != flist[i].fname && isdigit(*ptr) && isdigit(*(ptr-1))) --ptr;
	      }
	   else {
  	      Print("Invalid argument to -renumber : Aborted\n");
	      exit(1);
	      }

	   // p2 points to the first character past this numeric section
	   if (!isdigit(*ptr)) {
		Print("Filename '%s' has no numeric portion, renumbering cancelled\n",flist[i].fname);
		exit(1);
		}

	   p2 = ptr; while(isdigit(*p2)) ++p2;

	   if (p2 <= ptr) {
		Print("Filename '%s' does not start with digits. Cancelled\n",flist[i].fname);
		return;
		}

	   j = ptr - flist[i].fname; // number of bytes to copy from original name
	   
	   if (flist_idx <= 100) ndigits=2;
	   else if (flist_idx<=1000) ndigits=3;
	   else if (flist_idx<=10000) ndigits=4;
	   else if (flist_idx<=100000) ndigits=5;
	   else ndigits=6;  // This is not likely to happen

	   newname = (char *)malloc(strlen(flist[i].fname)+10); // make space for longer filename
	   strncpy(newname,flist[i].fname,j);
	   sprintf(newname+j,"q%0*d%s",ndigits,count,p2);

	   flist[i].newname = newname;
	   ++count;
	   }

	// No errors, so now proceed with the renumbering
	if (!Quiet) Print("\tRenaming %d files...\n",flist_idx);
	for(i=0; i<flist_idx; ++i) {
	   char *old_name = flist[i].fname;
	   char *new_name = flist[i].newname;

	   if (rename(old_name,new_name)) {
		Print("Rename '%s' -> '%s' failed with error: ",old_name,new_name);
		perror(NULL);
		exit(1);
		}
	   }

	if (!Quiet) Print("\tdone.\n");
	return;
	}

void
QAssociate(char *fname, double quality)
	{
	char pfx[256],sfx[32];
	int n,bucket;

	if (flist_idx == MAX_FILES) {
	   Print("QAssociate: Max file count of %d reached\n",MAX_FILES);
	   exit(1);
	   }
	flist[flist_idx].fname = strdup(fname);
	flist[flist_idx].val = quality;

	// Look for filename like NNN-D.fit, D is the bucket number
	if (sscanf(fname,"%s-%1d.%s",pfx,&n,sfx) == 3)
	   bucket = n;
	else 
	   bucket=0;
	
	flist[flist_idx].bucket = bucket;
	flist_idx++;
	}

/*
 * Generate a series of downsampled images and add up their quality
 * 
 * We handle FITS monochrome files only, either 16bpp or 8bpp. Process the region
 * represented by (x1,y1) to (x2,y2)
 */

double
QualityEstimate(unsigned char *buffer, int width, int height, int depth,
	int x1, int y1, int x2, int y2)
	{
	int subsample;
	int i,bpp,n,x,y,nbytes,max,x_inc;
	int x_samples,y_samples,y_last;
	int pixels_per_row, nrows;
	unsigned short *ubuffer,*buf,val;
	double mult,q,dval=0.0;
	int threshhold = ThreshHold;

	if (depth != 16 && depth != 8 && depth != 24) {
	   Print("QualityEstimate: Depth must be 8,16 or 24, not %d\n",depth);
	   exit(1);
	   }

	if (depth == 16) threshhold <<= 8;

	Clip(width,height,&x1,&y1,&x2,&y2);

	pixels_per_row = x2-x1+1;
	nrows = y2-y1+1;
	bpp = depth/8;

	// Allocate the intermediate buffer. Will be 16bpp greyscale
	buf = (unsigned short *)malloc(pixels_per_row * nrows * 2);
	if (buf == NULL) {
	  Print("QualityEstimate: Out of memory");
	  exit(1);
	  }

	subsample = QSUBSAMPLE_MIN;
	while(subsample <= QSUBSAMPLE_MAX) {
	   unsigned char *ptr;

	   /*
	    * Number of h & v pixels in subimage
	    */
	   x_samples = pixels_per_row / subsample;
	   y_samples = nrows / subsample;

	   if (x_samples < 2 || y_samples < 2) break;

	   max=0;
	   y_last = y2 - subsample + 1;
	   x_inc = subsample * bpp;

	   // First row - ignore histo-stretch
	   y=y1; n=0;
	   ptr = buffer + (y*width + x1) * bpp;
	   for(x=0; x < x_samples; ++x, ptr += x_inc) {
		  buf[n] = SubSample(ptr,width,bpp,subsample);
		  if (buf[n] < threshhold) buf[n]=0; // don't histo-stretch background
		  n++;
		  }

	   // Rows 1 .. y_last-1 additional histo-stretch code
	   for(y+=subsample; y < y_last; y += subsample) {
	      ptr = buffer + (y*width + x1) * bpp;
	      for(x=0; x < x_samples; ++x, ptr += x_inc) {
		  buf[n] = SubSample(ptr,width,bpp,subsample);
		  if (buf[n] < threshhold) buf[n]=0; // don't histo-stretch background
		  if (buf[n]>max) max=buf[n];
		  n++;
		  }
	     }

	   // Last row, ignore histo-stretch
	   ptr = buffer + (y*width + x1) * bpp;
	   for(x=0; x < x_samples; ++x, ptr += x_inc) {
		  buf[n] = SubSample(ptr,width,bpp,subsample);
		  if (buf[n] < threshhold) buf[n]=0; // don't histo-stretch background
		  n++;
		  }

	   // Stretch histogram
	   if (max>0) {
	      mult = (double)(256*256-1) / (double)max;
	      for(i=0; i<n; ++i) {
		unsigned int v = buf[i]; v *= mult;
		if (v >= 256*256) v = 256*256-1;
		buf[i] = v;
		}
	      }

	   /* Write this image (debugging) */
	   if (QWriteIntermediateFiles) {
	      char filename[1024];
	      FILE *out;

	      sprintf(filename,"sample_%d.ppm",subsample);
	      out = fopen(filename,"wb");
	      if (out == NULL) {
		Print("Cannot write subsampled image %d\n",subsample);
		exit(1);
		}

	      WritePGMHeader8(out,x_samples,y_samples);
	      for(i=0; i<n; ++i) putc(buf[i]>>8,out);

	      fclose(out);
	      Print("[%d:%s] ",subsample,filename);
	      }

	   q = AnalyseImage16(buf,x_samples,y_samples);
	   if (!Quiet) Print("%d:%2.2lf ",subsample,q);
	   dval += q;

	   do { subsample ++; }
	   while (width/subsample == x_samples && height/subsample == y_samples);
	   }

	free(buf);
	return dval;
	}

	   
/*
 * Subsample a region starting at *ptr of size X size pixels.
 * Always return a 16bpp result
 */
static int
SubSample(unsigned char *ptr, int img_wid, int bpp, int size)
	{
	int x,y,val=0;
	unsigned short v,*uptr;
	double d;
	
	switch(bpp) {
	   case 1:
		for(y=0; y<size; ++y) {
	   	   for(x=0; x<size; ++x) val += ((unsigned short)ptr[x]) << 8;
		   ptr += img_wid;
	   	   }
		break;

	   case 2:
		uptr = (unsigned short *)ptr;
		for(y=0; y<size; ++y) {
	   	   for(x=0; x<size; ++x) val += uptr[x];
		   uptr += img_wid;
	   	   }
		break;

	   case 3:
		// 24bpp data, 8 bit BGR, convert to monochrome
		for(y=0; y<size; ++y) {
	   	   for(x=0; x<size*3; x+=3) {
			d = ((double)ptr[x])*0.114;
			d += ((double)ptr[x+1])*0.587;
			d += ((double)ptr[x+2])*0.299;
			val += ((unsigned short)(d))<<8;
			}
		   ptr += img_wid*3;
	   	   }
		break;

	   default:
		Print("SubSample: Unsupported depth %d\n",bpp);
		exit(1);
	   }

	return val / (size * size);
	}

static double
AnalyseImage16(unsigned short *buf, int width, int height)
	{
	unsigned int x,y;
	int zerospan=0;
	int pixels=0;
	double val=0,d1,d2;
	int threshhold = ThreshHold << 8;

	/*
	 * Scan left -> right, top -> bottom. For each pixel calculate the difference
	 * to (x+1,y) and (x,y+1)
	 */

	for(y=0; y < height-1; ++y) {
	   unsigned short *lptr = buf + y * width;  	// start of row y
	   unsigned short *dptr = lptr + width;		// start of row (y+1)
	   zerospan=0;

	   for(x=0; x < width-1; ++x,++lptr,++dptr) {
		/* Skip if we find a span of black (background) */
		if (*lptr < threshhold) ++zerospan; else zerospan=0;
		if (zerospan > 2) continue;

		// Pixel differences weighted (Luminance,Colour)
		d1 = (*lptr - *(lptr+1));
		d2 = (*lptr - *dptr);
		val += d1*d1 + d2*d2;
		pixels++;
		}
	   }

	// return (total_differences) / (pixels processed)
	if (pixels==0) return 0;
	val /= (double)pixels * 65536 * 256;
	return val;
	}

/*
 * Distance in colourspace between the pixels, normalised
 */
static double
PixDiff(unsigned char *p1, unsigned char *p2)
	{
	int r = *(p1++) - *(p2++);
	int g = *(p1++) - *(p2++);
	int b = *(p1++) - *(p2++);

	return (r*r) + (g*g) + (b*b) / 1966080.0;
	}

/*
 * Distance in colourspace between the pixels
 */
inline double
PixDiffLC1(unsigned char *p1, unsigned char *p2, int depth, double l, double c)
	{
	int b,g,r;

	if (depth==24) {
	   b = *(p1++) - *(p2++);	// 0.0 - 255.0
	   g = *(p1++) - *(p2++);	// 0.0 - 255.0
	   r = *(p1++) - *(p2++);	// 0.0 - 255.0
	   double L,C;

	   // square to exaggerate differences
	   r *= r; g *= g; b *= b;

	   // Luminance difference (square again)
	   L = r * 0.299 + g * 0.587 + b * 0.114;
	   L *= L;

	   // Colourspace difference (square again)
	   C = r + g + b;
	   C *= C;

	   // Weighted result, reduced
	   return (L * l + C * c) / (195075.0 * (l+c));
	   }

	if (depth==8) {
	   b = *(p1++) - *(p2++);
	   return ((double)b * (double)b) / 65025.0;
	   }
	}
