#include "ppmcentre.h"

// Detect bright or dark pixels and replace them. We assume that
// most noise causes pixels to become brighter, so we err on the side
// of darkness when replacing a bright pixel.

static int smooth_image_16(unsigned char *data,int width,int height,int scale);

static unsigned int filter_upixel(register unsigned char *p, register int w, int *changed);
static unsigned short filter_ipixel(register unsigned short *p, register int w, int *changed);
static unsigned int filter_cpixel(register unsigned char *p, register int w, unsigned char rgb[3]);

#define BLACK_RGB(d) (*(d)=0,*((d)+1)=0,*((d)+2)=0)
#define WHITE_RGB(d) (*(d)=255,*((d)+1)=255,*((d)+2)=255)

#define SET_RGB(d,s) (*(d)=*(s),*((d)+1)=*((s)+1),*((d)+2)=*((s)+2))
#define ADD_RGB(d,s) (*(d)+=*(s),*((d)+1)+=*((s)+1),*((d)+2)+=*((s)+2))
#define LUM_RGB(p)   (*(p)*0.299 + *((p)+1)*0.587 + *((p)+2)*0.114)

static unsigned int
filter_upixel(register unsigned char *p, register int w, int *changed)
	{
	int min,max;
	int d,val,avg;
	double diff;
	int s1,s2,s3,s4,s5,s6,s7,s8,cp;

	p -= (w+1);
	s1 = *(p++); s2 = *(p++); s3 = *p; p += w;
	s4 = *(p--); cp = *(p--); s5 = *p; p+=w;
	s6 = *(p++); s7 = *(p++); s8 = *p;

	min=s1; max=min;
	if (s2 < min) min=s2; if (s2 > max) max=s2;
	if (s3 < min) min=s3; if (s3 > max) max=s3;
	if (s4 < min) min=s4; if (s4 > max) max=s4;
	if (s5 < min) min=s5; if (s5 > max) max=s5;
	if (s6 < min) min=s6; if (s6 > max) max=s6;
	if (s7 < min) min=s7; if (s7 > max) max=s7;
	if (s8 < min) min=s8; if (s8 > max) max=s8;

	// Calculate modified average of neighbors,
	// discard brightest and darkest pixels pixel
	avg = s1+s2+s3+s4+s5+s6+s7+s8 - max - min;
	avg /= 6;

	// Hot pixel
	if (cp-avg > 50) {
	   *changed=1;
	   return avg;				// light pixel
	   }

	if (avg>0) diff = (double)(cp - avg) / (double)avg;
	else 	   diff = cp - avg;

	if (diff < -InputFilter_ThreshHold) {
		*changed=1;
		return (cp + cp + avg) / 3;		// dark pixel
		}
	if (diff < InputFilter_ThreshHold) {
		*changed=0;
		return cp;			// ok pixel
		}

	*changed=1;
	return avg;				// light pixel
	}

// Data is RGB, 3 bytes per pixel
// "w" is bytes per line in the image
// THIS CODE DOESN'T WORK YET
static unsigned int
filter_cpixel(register unsigned char *p, register int w, unsigned char rgb[3])
	{
	int i,val;
	double diff;
	int cp[3],avg[3];
	double d,a,lum[9],lum_cp;

	p -= (w+3);
	SET_RGB(avg,p); lum[0]=LUM_RGB(p); p+=3;
	ADD_RGB(avg,p); lum[1]=LUM_RGB(p); p+=3;
	ADD_RGB(avg,p); lum[2]=LUM_RGB(p); p += w;
	ADD_RGB(avg,p); lum[3]=LUM_RGB(p); p-=3;

	SET_RGB(cp,p);  lum_cp=LUM_RGB(p); p-=3;

	ADD_RGB(avg,p); lum[4]=LUM_RGB(p); p+=w;
	ADD_RGB(avg,p); lum[5]=LUM_RGB(p); p+=3; 
	ADD_RGB(avg,p); lum[6]=LUM_RGB(p); p+=3;
	ADD_RGB(avg,p); lum[7]=LUM_RGB(p);

	// Calculate average
	for(i=a=0; i<8; ++i) a+=lum[i]; a/=8;

	// How different are we from our neighbors?
	d = lum_cp - a; 

	if (d<0) d=-d;
	diff = d / a;

	if (diff >= InputFilter_ThreshHold) {
		*rgb = avg[0]/8; *(rgb+1)=avg[1]/8; *(rgb+2)=avg[2]/8;
		return(1);
		}

	return(0);
	}

static unsigned short
filter_ipixel(register unsigned short *p, register int w, int *changed)
	{
	int min,max;
	int val,avg;
	double diff;
	int s1,s2,s3,s4,s5,s6,s7,s8,cp;

	p -= (w+1);
	s1 = *(p++); s2 = *(p++); s3 = *p; p += w;
	s4 = *(p--); cp = *(p--); s5 = *p; p+=w;
	s6 = *(p++); s7 = *(p++); s8 = *p;

	min=s1; max=min;
	if (s2 < min) min=s2; if (s2 > max) max=s2;
	if (s3 < min) min=s3; if (s3 > max) max=s3;
	if (s4 < min) min=s4; if (s4 > max) max=s4;
	if (s5 < min) min=s5; if (s5 > max) max=s5;
	if (s6 < min) min=s6; if (s6 > max) max=s6;
	if (s7 < min) min=s7; if (s7 > max) max=s7;
	if (s8 < min) min=s8; if (s8 > max) max=s8;

	// Calculate modified average of neighbors,
	// discard brightest and darkest pixels pixel
	avg = s1+s2+s3+s4+s5+s6+s7+s8 - max - min;
	avg /= 6;

	// Hot pixel
	if (cp-avg > 50 * 256) {
	   *changed=1;
	   return avg;				// light pixel
	   }

	// Asymmetric - assume darker pixels are ok, lighter pixels are bad
	// if (d<0) d=-d;

	if (avg>0) diff = (double)(cp - avg) / (double)avg;
	else 	   diff = cp - avg;

	if (diff < -InputFilter_ThreshHold) {
		*changed=1;
		return (cp + cp + avg) / 3;		// dark pixel
		}
	if (diff < InputFilter_ThreshHold) {
		*changed=0;
		return cp;			// ok pixel
		}

	*changed=1;
	return avg;				// light pixel
	}

// input buffer[] of image data
// (width,height) in pixels
// rectangular region (x1,y1) to (x2,y2) hold the image	
int
input_filter(unsigned char *buffer, int width,int height,int depth,int x1, int y1, int x2, int y2)
	{
	int i,x,y,o,os,od,filter_count=0,threshhold,changed;
	unsigned short *isrc,*idst;
	unsigned char *usrc,*udst;
	int w,h;
	static int old_depth=-1,old_w=-1,old_h=-1;
	static unsigned char *tmpbuf = NULL;

	if (x1<0) x1=0; if (y1<0) y1=0;
	if (x2>=width) x2=width=1;
	if (y2>=height) y2=height=1;

	w = x2 - x1 + 1;
	h = y2 - y1 + 1;

	if (tmpbuf==NULL || old_w != w || old_h != h || old_depth != depth) {
	   if (tmpbuf) free(tmpbuf);

	   tmpbuf = (unsigned char *)malloc(w * h * depth/8);
	   if (tmpbuf==NULL) {
		Print("input_filter: out of memory\n");
		exit(1);
		}
	   old_depth = depth;
	   old_w = w; old_h = h;
	   }

	switch(depth) {
	   case 8:
			usrc = (unsigned char *) buffer;
			udst = (unsigned char *) tmpbuf;

			// set the threshhold so we only consider
			// pixels greater than 5% in brightness
			threshhold = 256 / 20;

			// copy row 0 with no filtering
			o = y1 * width + x1;
			memcpy(udst,usrc+o,w);

			// filter rows 1 .. height-2
			for(y=y1+1; y<y2; ++y) {
			  os = y * width + x1;  // offset into source
			  od = (y-y1) * w;  // offset into destination

			  udst[od] = usrc[os]; ++od; ++os;
			  for(x=x1+1; x<x2; ++x,++os,++od)
				if (usrc[os] < threshhold) udst[od]=usrc[os];
				else {
				   udst[od] = filter_upixel(usrc+os,width,&changed);
				   filter_count += changed;
				   }
			  udst[od] = usrc[os];
			  }
			
			// copy last row with no filtering
			os = y * width + x1;  // offset into source
			od = (y-y1) * w;  // offset into destination
			memcpy(udst+od,usrc+os,w);

			// copy it back
			os=0;
			od = y1 * width + x1;
			for(y=0; y<h; ++y,os+=w,od+=width) {
			   memcpy(usrc+od,udst+os,w);
			   }

			break;

	   case 16:
			isrc = (unsigned short *) buffer;
			idst = (unsigned short *) tmpbuf;

			// set the threshhold so we only consider
			// pixels greater than 5% in brightness
			threshhold = 63353 / 20;

			// copy row 0 with no filtering
			//o = y1 * width + x1;
			//memcpy(idst,isrc+o,w * 2);
		        for(i=0; i<w; ++i) idst[i]=0;

			// filter rows 1 .. height-2
			for(y=y1+1; y<y2; ++y) {
			  os = y * width + x1;  // offset into source
			  od = (y-y1) * w;  // offset into destination

			  idst[od++] = 0;
			  //idst[od] = isrc[os]; 
			  ++os;
			  for(x=x1+1; x<x2; ++x,++os,++od)
				if (isrc[os] < threshhold) idst[od]=isrc[os];
				else {
				   idst[od] = filter_ipixel(isrc+os,width,&changed);
				   filter_count += changed;
				   }
			  //idst[od] = isrc[os];
			  idst[od] = 0;
			  }
			
			os = y * width + x1;  // offset into source
			od = (y-y1) * w;  // offset into destination
			// copy last row with no filtering
			//memcpy(idst+od,isrc+os,w * 2);

			// Set the last row to black
		        for(i=0; i<w; ++i) idst[od+i]=0;

			// copy it back
			os=0;
			od = y1 * width + x1;
			for(y=0; y<h; ++y,os+=w,od+=width) {
			   memcpy(isrc+od,idst+os,w*2);
			   }

			break;

	   case 24:
			usrc = (unsigned char *) buffer;
			udst = (unsigned char *) tmpbuf;

			// set the threshhold so we only consider
			// pixels greater than 5% in brightness
			threshhold = 256 / 20;

			// copy row 0 with no filtering
			o = (y1 * width + x1) * 3;
			memcpy(udst,usrc+o,w * 3);

			// filter rows 1 .. height-2
			for(y=y1+1; y<y2; ++y) {
			  os = (y * width + x1) * 3;  // offset into source
			  od = (y-y1) * w * 3;  // offset into destination

			  SET_RGB(udst+od,usrc+os); od+=3; os+=3;
			  for(x=x1+1; x<x2; ++x,os+=3,od+=3)
				if (LUM_RGB(usrc+os) < threshhold) SET_RGB(udst+od,usrc+os);
				else {
				   unsigned char p[3];
				   if (filter_cpixel(usrc+os,width*3,p)) {
				      SET_RGB(udst+od,p); ++filter_count;
				      }
				   else {
				      SET_RGB(udst+od,usrc+os);
				      }
				   }
			  SET_RGB(udst+od,usrc+os);
			  }
			
			// copy last row with no filtering
			os = (y * width + x1)*3;  // offset into source
			od = (y-y1) * w * 3;  // offset into destination
			memcpy(udst+od,usrc+os,w * 3);

			// copy it back
			os=0;
			od = (y1 * width + x1) * 3;
			for(y=0; y<h; ++y,os+=w*3,od+=width*3) {
			   memcpy(usrc+od,udst+os,w*3);
			   }

			break;
		}

	return filter_count;
	}

// Generate the straight 3x3 average, but weight the centre pixel
static unsigned short
average_small_ipixel(register unsigned short *p, register int w)
	{
	int s1,s2,s3,s4,s5,s6,s7,s8,cp;

	p -= (w+1);
	s1 = *(p++); s2 = *(p++); s3 = *p; p += w;
	s4 = *(p--); cp = *(p--); s5 = *p; p+=w;
	s6 = *(p++); s7 = *(p++); s8 = *p;

	return (s1+s2+s3+s4+s5+s6+s7+s8+cp+cp) / 10;
	}

// p points to top-left corner of a large pixel that is sXs in size.
// buf is a buffer allocated by our caller that we can use for storing our
// intermediate results
// rowlen is the number of pixels in each row.
// s is the scale factor.

static void
smooth_large_ipixel(unsigned short *dst,unsigned short *src, int rowlen, int s)
	{
	int n,i,first=1;
	unsigned short *src_top,*src_bottom,*src_right;
	unsigned short *dst_bottom,*dst_right;
	unsigned short pixbuf[100];

	src_top = src;

	while(s>1) {
	   src_bottom = src_top + rowlen * (s-1);
	   src_right = src_top + s - 1;

	   dst_bottom = dst + rowlen * (s-1);
	   dst_right = dst + s - 1;

	   // horizontal pass (top and bottom)
	   for(i=n=0; i<s; ++i) {
		pixbuf[n++] = average_small_ipixel(src_top+i,rowlen);
		pixbuf[n++] = average_small_ipixel(src_bottom+i,rowlen);
		}

	   // vertical pass (left and right)
	   for(i=1; i<s-1; ++i) {
		pixbuf[n++] = average_small_ipixel(src_top+rowlen*i,rowlen);
		pixbuf[n++] = average_small_ipixel(src_right + rowlen*i,rowlen);
		}

	   // Copy the pixels to the destination
	   for(i=n=0; i<s; ++i) {
		*(dst+i) = pixbuf[n++];
		*(dst_bottom+i) = pixbuf[n++];
		}
	   for(i=1; i<s-1; ++i) {
		*(dst+rowlen*i) = pixbuf[n++];
		*(dst_right+rowlen*i) = pixbuf[n++];
		}

	   // Walk inward toward centre of large pixel (down and right) and repeat
	   s-=2; dst += rowlen+1;

	   // subsequent passes use dst as the source so we get a real blending
	   src_top = dst;
	   }
	}

// Input is an image that has just been upscaled by a factor of "scale".
// Each original pixel has been replaced by a scaleXscale square block of pixels
// Now we work out how to smooth this block to reduce edge artifacts
// Input width and height refer to the upscaled image, not the original image

int
smooth_image(unsigned char *data,int width,int height,int depth,int scale)
	{
	if (scale<=2) return(0);

	switch(depth) {
	   case 8:
		Print("smoothing not implemented for depth 8\n");
		exit(1);
		break;
	   case 16:
		return smooth_image_16(data,width,height,scale);
		break;
	   case 24:
		Print("smoothing not implemented for depth 24\n");
		exit(1);
		break;
	   default:
		Print("smoothing not implemented for unknown depth %d\n",depth);
		exit(1);
		break;
	   }
	return(0);
	}

static int
smooth_image_16(unsigned char *data,int width,int height,int scale)
	{
	int x,y,buffer_size,large_width,large_height;
	static unsigned char *buffer = NULL;
	static int old_buffer_size = -1;
	unsigned short *isrc,*ibuf;

	buffer_size = width * height * 2;

	if (buffer == NULL || old_buffer_size != buffer_size) {
	   if (buffer) free(buffer);
	   buffer = (unsigned char *)malloc(buffer_size);
	   if (buffer==NULL) {
		Print("Out of memory allocating buffer in smooth_image\n");
		exit(1);
		}
	   old_buffer_size = buffer_size;
	   }


	// Image dimensions in "large pixels" 
	large_height = height / scale;
	large_width = width / scale;

	// For ODD scaling values we use a simple 3x3 mean function
	// working clockwise around each large pixel, from edge to centre.
	// Leave the centre pixel alone.
	if (scale & 1) {
	   int o,x1,x2,y1,y2;
	   int v_inc = width * (scale-1);

	   // Start at top left corner of large pixel at (1,1)

	   y1 = (large_height - CutY)/2 - 1; if (y1<0) y1=0;
	   y2 = y1 + CutY+2; if (y2>large_height) y2=large_height;
	   x1 = (large_width - CutX)/2 -1; if (x1<0) x1=0;
	   x2 = x1 + CutX+2; if (x2>large_width) x2=large_width;

	   y1 *= scale; y2 *= scale;
	   x1 *= scale; x2 *= scale;

	   // Copy the src image into the dst to help in our
	   // smooth_large_ipixel routine
	   for(y=y1; y<y2; ++y) {
	     o = y * width;
	     isrc = (unsigned short *)data + o;
	     ibuf = (unsigned short *)buffer + o;
	     for(x=x1; x<x2; ++x) ibuf[x] = isrc[x];
	     }

	   y1 = (large_height - CutY)/2; if (y1<1) y1=1;
	   y2 = y1 + CutY; if (y2>large_height-2) y2=large_height-2;
	   x1 = (large_width - CutX)/2; if (x1<1) x1=1;
	   x2 = x1 + CutX; if (x2>large_width-2) x2=large_width-2;

	   for(y = y1; y <= y2; ++y) {
	     o = width * scale * y;
	     isrc = (unsigned short *)data + o;
	     ibuf = (unsigned short *)buffer + o;
	     for(x=x1; x<=x2; ++x)
		smooth_large_ipixel(isrc+x*scale,ibuf+x*scale,width,scale);
	     }

	   }
	else {
	   Print("smoothing requested, but not possible with even upscaling factor\n");
	   exit(0);
	   }

	return(0);
	}

// Normal Average
#define AVERAGE(a,b) (((a)+(b))>>1)

// weighted average, favours the dark side by 25%
#define DARK_AVERAGE(a,b) ((a)>(b)?((a)*3+(b)*5)>>3:((b)*3+(a)*5)>>3)

// scale existing widthxheight image by factor n
// in-place. Work from bottom->top right->left duplicating pixel data.
// width & height refer to the src image dimensions.
int
upscale_image(unsigned char *data,int *width,int *height, int bpp, int n)
	{
	int i,j,k,l,x,y;
	unsigned char *src,*dst;
	unsigned short *isrc,*idst;
	int rowbytes = (*width) * bpp;
	int imagebytes = rowbytes * (*height);

	src = data + imagebytes - bpp;		  // last pixel in input image
	dst = data + (imagebytes * n * n) - bpp;    // last pixel in output image

	switch(bpp) {
	   case 1:
		for(y=(*height)-1; y>=0; --y,src-=rowbytes)
	   	   // Duplicate each input row n times, reset src position
	   	   // each time to end of current input row
	   	   for(j=0; j<n; ++j,src+=rowbytes)
	      		// copy a row with expansion
	      		for(x=rowbytes-1; x>=0; --x,--src)
			   for(i=0; i<n; ++i) *(dst--) = *src;
		break;

	   case 2:
		isrc = (unsigned short *) src;
		idst = (unsigned short *) dst;

		for(y=(*height)-1; y>=0; --y, isrc-=*width) {
	   	   // Duplicate each input row n times, reset src position
	   	   // each time to end of current input row

	   	   for(j=0; j<n; ++j,isrc+=*width)
			// copy 1 pixel in row
	      		for(x=(*width)-1; x>=0; --x,--isrc)
			   for(i=0; i<n; ++i) *(idst--) = *isrc;
		   }

		break;
	   case 3:
		for(y=(*height)-1; y>=0; --y,src-=rowbytes)
	   	   // Duplicate each input row n times, reset src position
	   	   // each time to end of current input row
	   	   for(j=0; j<n; ++j,src+=rowbytes)
	      		// copy a row with expansion
	      		for(x=rowbytes-1; x>=0; x-=bpp,src-=bpp)
			   for(i=0; i<n; ++i,dst-=bpp) {
				*dst = *src; *(dst+1) = *(src+1); *(dst+2) = *(src+2); 
				}
		break;
	   }

	*width  *= UpScale;
	*height *= UpScale;
	return(0);
	}

int
downscale_image(unsigned char *data,int *width,int *height, int bpp, int n)
	{
	int i,j,o,x,y;
	unsigned char *src,*dst;
	unsigned short *isrc,*idst;
	int rowbytes = (*width) * bpp;
	int nw = (*width)/n;
	int nh = (*height)/n;
	unsigned char *rowbuffer = NULL;
	unsigned int *rowbuffer_i = NULL;
	unsigned int *rowbuffer_r = NULL;
	unsigned int *rowbuffer_b = NULL;
	unsigned int *rowbuffer_g = NULL;
	int total,total_r,total_g,total_b;
	int sample_size = n * n;

	src = data;   // first pixel in input image
	dst = data;   // last pixel in output image

	// We are shrinking the image, so work left->right and top->bottom
	// so we don't overwrite needed data

	switch(bpp) {
	   case 1:
		// Monochrome 8bpp
		rowbuffer_i = (unsigned int *)malloc(nw * sizeof(unsigned int));

		for(y=0; y<(*height)-n; y+=n) {
		   // Load the first row totals
		   for(o=x=0; x<(*width)-n; x+=n) {
			for(j=total=0; j<n; ++j) total += *(src++);
			rowbuffer_i[o++] = total;
			}
		   while(x++ < *width) ++src;  // take care of alignment issues

		   // Add more totals as required for successive input rows
		   for(i=1; i<n; ++i) {
		     for(o=x=0; x<(*width)-n; x+=n) {
			for(j=total=0; j<n; ++j) total += *(src++);
			rowbuffer_i[o++] += total;
			}
		      while(x++ < *width) ++src;  // take care of alignment issues
		      }

		   // Output averages
		   for(x=0; x<nw; ++x) *(dst++) = rowbuffer_i[x] / sample_size;
		   }
		free(rowbuffer_i);
		break;

	   case 3:
		// RGB 24bpp
		rowbuffer_r = (unsigned int *)malloc(nw * sizeof(unsigned int));
		rowbuffer_g = (unsigned int *)malloc(nw * sizeof(unsigned int));
		rowbuffer_b = (unsigned int *)malloc(nw * sizeof(unsigned int));

		for(y=0; y<(*height)-n; y+=n) {
		   // Load the first row totals
		   for(o=x=0; x<(*width)-n; x+=n) {
			for(j=total_r=total_g=total_b=0; j<n; ++j)
			  {total_r += *(src++); total_g += *(src++); total_b += *(src++);}
			rowbuffer_r[o] = total_r;
			rowbuffer_g[o] = total_g;
			rowbuffer_b[o] = total_b; ++o;
			}
		   while(x++ < *width) src+=3;  // take care of alignment issues

		   // Add more totals as required for successive input rows
		   for(i=1; i<n; ++i) {
		     for(o=x=0; x<(*width)-n; x+=n) {
			for(j=total_r=total_g=total_b=0; j<n; ++j)
			  {total_r += *(src++); total_g += *(src++); total_b += *(src++);}
			rowbuffer_r[o] += total_r;
			rowbuffer_g[o] += total_g;
			rowbuffer_b[o] += total_b; ++o;
			}
		      while(x++ < *width) src+=3;  // take care of alignment issues
		      }

		   // Output averages
		   for(x=0; x<nw; ++x) {
			*(dst++) = rowbuffer_r[x] / sample_size;
			*(dst++) = rowbuffer_g[x] / sample_size;
			*(dst++) = rowbuffer_b[x] / sample_size;
			}
		   }

		free(rowbuffer_r);
		free(rowbuffer_g);
		free(rowbuffer_b);
		break;

	   case 2:
		// Monochrome 16bpp
		isrc = (unsigned short *)data;
		idst = (unsigned short *)data;

		rowbuffer_i = (unsigned int *)malloc(nw * sizeof(unsigned int));

		for(y=0; y<(*height)-n; y+=n) {
		   // Load the first row totals
		   for(o=x=0; x<(*width)-n; x+=n) {
			for(j=total=0; j<n; ++j) total += *(isrc++);
			rowbuffer_i[o++] = total;
			}
		   while(x++ < *width) ++isrc;  // take care of alignment issues

		   // Add more totals as required for successive input rows
		   for(i=1; i<n; ++i) {
		     for(o=x=0; x<(*width)-n; x+=n) {
			for(j=total=0; j<n; ++j) total += *(isrc++);
			rowbuffer_i[o++] += total;
			}
		      while(x++ < *width) ++isrc;  // take care of alignment issues
		      }

		   // Output averages
		   for(x=0; x<nw; ++x) *(idst++) = rowbuffer_i[x] / sample_size;
		   }
		free(rowbuffer_i);
		break;
	   }

	*width = nw;
	*height = nh;

	return(0);
	}
