#include "ppmcentre.h"

static int AlignBMP(char *fname, char *out_fname);
static int AlignFIT(char *fname, char *out_fname);

static void
CheckDir(char *dirname)
	{
        if (! isDirectory(dirname)) {
#ifdef MSWIN32
             mkdir(dirname);
#else
             mkdir(dirname,0755);
#endif

            if (! isDirectory(dirname)) {
               Print("Cannot create output directory %s\n",dirname);
               exit(1);
               }
	   }
	}

int Align(char *src_fname)
	{
	char *p,*dst_fname = src_fname;
	int need_to_free = 0;
	int i,err;

	// Canonise filename
	for(i=0; src_fname[i]; ++i)
	   if (src_fname[i]=='\\') src_fname[i]='/';

	// Get rid of leading "./"
	if (src_fname[0]=='.' && src_fname[1]=='/') {
	   for(i=2; src_fname[i]; ++i) src_fname[i-2]=src_fname[i];
	   src_fname[i-2]=0;
	   }
	
	// If -outdir option is given write the files to the named directory
	// instead of overwriting the original files. OutDir is relative to
	// the source file directory
	if (OutDir) {
	   char *strtmp = strdup(src_fname);
	   char *ptr = strtmp + strlen(strtmp) -1;
	   int num;
	   char sfx[10],type[10],*pfx_dir;

	   while(ptr != strtmp && *ptr != '/') --ptr;
	   if (ptr != strtmp) {
		*(ptr++)=0;
		pfx_dir = strtmp;
		}
	   else {
		pfx_dir=".";
		}

	   // Look for filename "dddddd-X.sss" d=digits, sss=suffix, X is
	   // identifier. If found, create subdirectory "X"
	   if (sscanf(ptr,"%d-%1s.%s",&num,type,sfx) != 3) 
		type[0]=0;

	   dst_fname = malloc(strlen(pfx_dir) + strlen(OutDir) + 
			strlen(type) + strlen(ptr) + 4);
	   if (! dst_fname) {
		Print("Align: Out of Memory\n");
		exit(1);
		}

	    // If OutDir is absolute, then just use it, else if it is relative
	    // then prepend our directory prefix
	    if (OutDir[0]=='/')
		strcpy(dst_fname,OutDir);
	    else
	    	sprintf(dst_fname,"%s/%s",pfx_dir,OutDir);

	    CheckDir(dst_fname);

	     if (type[0]) {
		strcat(dst_fname,"/");
		strcat(dst_fname,type);
		CheckDir(dst_fname);
		if (forceWriteEmptyFiles) 
		     WriteEmptyFiles = 1;
		else WriteEmptyFiles=0;
		}

	   strcat(dst_fname,"/");
	   strcat(dst_fname,ptr);
	   need_to_free = 1;
	   free(strtmp);
	   }

	if (NoSave==0 && AllowOverwrite==0 && !access(dst_fname,R_OK)) {
	   if (!Quiet) Print("Not overwriting output file %s\n",dst_fname);
	   err=1; goto end;
	   }

	p = src_fname;
	while(*p) ++p; --p;
	while(p != src_fname && *(p-1) != '/') --p;
	CurrentFile = strdup(p);

	// Dispatch based on file suffix
	p = src_fname;
	while(*p) ++p; --p;
	while(p != src_fname && *p != '.') --p;

	if (! strcasecmp(p,".bmp")) err = AlignBMP(src_fname,dst_fname);
	else if (! strcasecmp(p,".fit")) err =  AlignFIT(src_fname,dst_fname);
	else if (! strcasecmp(p,".fts")) err =  AlignFIT(src_fname,dst_fname);
	else if (! strcasecmp(p,".fits")) err =  AlignFIT(src_fname,dst_fname);
	else {
	   Print("Unknown file suffix for '%s'\n",src_fname);
	   exit(1);
	   }
	Print("\n");

	end:
	free(CurrentFile); CurrentFile=NULL;
	if (need_to_free) free(dst_fname);

	return err;
	}

static int
AlignBMP(char *fname, char *out_fname)
	{
	FILE *in = NULL;
	FILE *out = NULL;
	int width,height,depth,colourmax;
	char type[10];
	unsigned char map[8192];		/* colour map buffer */
	unsigned char rmap[8192];	/* maps colour -> colourmap (greyscale only) */
	int entries;		/* Number of colourmap entries */
	int x,y,x1,y1,x2,y2,i,n,err;
	int nw,nh,bufsize = 0;
	int tmpbufsize;
	unsigned char *tmpbuf,*sptr,*dptr;
	unsigned char *buffer,*outbuffer;
	bmp_header H;

	in = fopen(fname,"rb");
	if (in == NULL) {
	   fprintf(stderr,"ppmcenter: cannot open input file\n");
	   Usage();
	   exit(1);
	   }

	/*
	 * Read the input
	 */
	type[0] = 0;
	entries=0;

	read_bmp_header(in,type,&width,&height,&depth,map,&entries);

	if (strcmp(type,"BM")) {
	   Print("Input is not a BMP\n");
	   exit(1);
	   }

	if (!Quiet) Print("(%dx%dx%d) ",width,height,depth);

	/*
	 * Allocate the buffer and load the data
	 */
	bufsize = width * height * 3;
	buffer = (unsigned char *)malloc(bufsize);
	if (buffer == NULL) {
		Print("Cannot malloc buffer of %d bytes\n",bufsize);
		exit(1);
		}

	/* If we have a colourmap, load the buffer and fill from the colourmap */
	if (entries > 0) {
	   tmpbufsize = width * height;
	   tmpbuf = (unsigned char *)malloc(bufsize);
	   if (tmpbuf == NULL) {
		Print("Cannot malloc buffer of %d bytes\n",tmpbufsize);
		exit(1);
		}

	   err = fread(tmpbuf,1,tmpbufsize,in);
	   if (err != tmpbufsize) {
	   	Print("Error in reading data, only read %d bytes of %d\n",err,tmpbufsize);
	   	exit(1);
	   	}

	   for(i=0,sptr=tmpbuf,dptr=buffer; i<tmpbufsize; ++sptr, ++i) {
		n = tmpbuf[i];
		if (n>=entries) {
		   Print("Colourmap entry %d > colourmap size of %d\n",n,entries);
		   exit(1);
		   }

		n *= 4;
		*(dptr++) = map[n++];
		*(dptr++) = map[n++];
		*(dptr++) = map[n++];
		}
	   free(tmpbuf);

	   /* create the reverse colourmap that we will use to write out the image later on */
	   for(i=0; i<entries; ++i) {
	 	n = map[i*4];
		rmap[n] = i;  /* it's grey, just use one channel of data */
		}
	   }
	else {
	   // Data is BGR
	   err = fread(buffer,1,bufsize,in);
	   if (err != bufsize) {
	      Print("Error in reading data, only read %d bytes of %d\n",err,bufsize);
	      exit(1);
	      }
	   }

	fclose(in);
	outbuffer = (unsigned char *)malloc(bufsize * UpScale * UpScale);
	if (outbuffer == NULL) {
		Print("Out of memory\n");
		exit(1);
		}

	/* If requested, debayer the image */
	if (WantDebayer) DeBayer(buffer,width,height);

	if (! process(&width,&height,24,buffer,outbuffer,out_fname)) {
		Print("[no output]");
		free(buffer);
		free(outbuffer);

		// Make sure the output file exists even if it is size 0
		if (access(out_fname,0) && WriteEmptyFiles) {
		   out = fopen(out_fname,"wb");
		   fclose(out);
		   }

		return(1);
		}

	// Now we switch to working at the output resolution. process()
	// has altered width and height for us.
	if (newWidth < 0) nw = width;
	else 		  nw = (newWidth * UpScale) / DownScale;

	if (newHeight < 0) nh = height;
	else 		   nh = (newHeight * UpScale) / DownScale;

	// Width must be a multiple of 4 bytes
	nw -= nw & 3;

	if (nw < 0) nw = width;
	if (nh < 0) nh = height;

	if (nw > width) nw = width;
	if (nh > height) nh = height;

	if (NoSave) {
	   Print(" [skipped]");
	   goto end;
	   }

	if (!Quiet) Print("-> %s (%dx%dx%d)\n",out_fname,nw,nh,depth);

	out = fopen(out_fname,"wb");
	if (! out) {
	   Print("Cannot open '%s' for writing\n",out_fname);
	   exit(1);
	   }

	if (! fwrite("BM",2,1,out)) {
	   Print("Short write on %s\n",out_fname);
	   exit(1);
	   }

	if (depth == 8) {
	   H.filesz = 54 + (nw * nh) + 1024;
	   H.offset = 54 + 1024;
	   H.bitcount = 8;
	   H.cmapentries = entries;
	   }
	else {
	   H.filesz = 54 + (nw * nh * 3);
	   H.offset = 54;
	   H.bitcount = 24;
	   H.cmapentries = 0;
	   }

	H.r1 = H.r2 = 0;
	H.size = 40;
	H.width = nw;
	H.height = nh;
	H.planes = 1;
	H.compression = 0;
	H.sizeimage = 0;
	H.xppm = 0;
	H.yppm = 0;
	H.clrimportant = 0;

	if (! fwrite((char *)&H,sizeof(H),1,out)) {
	   Print("cannot write to output file %s\n",out_fname);
	   exit(1);
	   }

	/* If we have a colourmap then write it out */
	if (entries>0) {
	   if (! fwrite(map,entries * 4,1,out)) {
	      Print("Cannot write colourmap to output file %s\n",out_fname);
	      exit(1);
	      }
	   }

	/* Now write the buffer according to our reverse colourmap */
	/* Write a window with dimensions (newWidth, newHeight) */
	y1 = (height - nh)/2; y2 = y1 + nh-1;
	x1 = (width - nw)/2; x2 = x1 + nw-1;
	   
	x1 *= 3; x2 *= 3;
	for(sptr = outbuffer + y1*width*3,y = y1; y<=y2; y++, sptr += width*3)
	   if (depth==8) for(x=x1; x<=x2; x+=3) { putc(rmap[sptr[x]],out); }
	   else {
		if (! fwrite(sptr+x1, nw*3, 1, out)) {
		   Print("Write failed to file %s\n",out_fname);
		   exit(1);
		   }
		}

	fclose(out);

	end:

	free(buffer);
	free(outbuffer);
	return(1);
	}

static int
AlignFIT(char *fname, char *out_fname)
	{
	FILE *in = NULL;
	FILE *out = NULL;
	double bscale = 1.0;
	int width,height,depth,nw,nh;
	int x,y,x1,y1,x2,y2,i,n,err;
	int bpp,bufsize = 0;
	unsigned char *sptr;
	static unsigned char *buffer = NULL;
	static unsigned char *outbuffer = NULL;
	static int outbuffer_size = -1;
	static int buffer_size = -1;
	int bzero = 0;
	short *iptr;

	in = fopen(fname,"rb");
	if (in == NULL) {
	   Print("ppmcenter: cannot open input file\n");
	   Usage();
	   exit(1);
	   }

	/*
	 * Read the header
	 */
	read_fits_header(in,&width,&height,&depth,&bscale,&bzero);
	bpp = depth/8;

	if (!Quiet) Print("(%dx%dx%d) ",width,height,depth);

	/*
	 * Allocate the buffer and load the data
	 */
	bufsize = width * height * bpp;

	if (buffer==NULL || buffer_size != bufsize) {
	   if (buffer) free(buffer);
	   buffer = (unsigned char *)malloc(bufsize);
	   if (buffer == NULL) {
		Print("Cannot malloc buffer of %d bytes\n",bufsize);
		exit(1);
		}
	   buffer_size = bufsize;
	   }

	err = fread(buffer,1,bufsize,in); fclose(in);
	if (err != bufsize) {
	      Print("Error in reading data, only read %d bytes of %d\n",err,bufsize);
	      exit(1);
	      }

	if (outbuffer == NULL || outbuffer_size != bufsize) {
	   if (outbuffer) free(outbuffer);
	   outbuffer_size = bufsize * UpScale * UpScale;
	   outbuffer = (unsigned char *)malloc(outbuffer_size);
	   if (outbuffer == NULL) {
		Print("Cannot malloc outbuffer of %d bytes\n",bufsize);
		exit(1);
		}
	   }

	// Undo the endian-ness of the data for 16bpp FITS data
	if (depth == 16) {
	   unsigned char *tmp,*ptr;
	   unsigned short *iptr;

	   ptr = buffer; iptr = (unsigned short *)ptr;
	   i=width * height;
	   while(i--) {
		x = *ptr; *ptr = *(ptr+1); *(ptr+1) = x;
		if (bzero) *iptr += bzero;
		//if (bscale != 1.0) *iptr *= bscale;
		ptr += 2; iptr++;
		}
	   }
	else if (depth == 8) {
	   // Unsigned 8 bit data
	   unsigned char *ptr = buffer;
	   i = width * height;
	   while(i--) {
		if (bzero) *ptr += bzero;
		//if (bscale != 1.0) *ptr *= bscale;
		ptr++;
		}
	   }
	else {
	   Print("AlignFIT: Unsupported depth: %d\n",depth);
	   exit(1);
	   }

	// Possibly merge data from a MergeFile
	if (MergeFile) {
	   merge_data(MergeFile,buffer,width,height,depth);
	   }

	// Possibly stack the input frame
	if (StackFile) {
	   if (StackCount < StackMax) {
	      stack_frame(buffer,width,height,depth);
	      StackCount++;
	      if (!Quiet) Print("[Stack %d] ",StackCount);
	      }
	   }

	if (NoSave) {
	   Print("[skipped]");
	   return(1);
	   }

	if (! process(&width,&height,depth,buffer,outbuffer,out_fname)) {
		Print("[no output] "); fflush(stderr);

		// Make sure the output file exists even if it is size 0
		if (access(out_fname,0) && WriteEmptyFiles) {
		   out = fopen(out_fname,"wb");
		   fclose(out);
		   }

		return(1);
		}

	if (newWidth < 0) nw = width;
	else 		  nw = (newWidth * UpScale) / DownScale;

	if (newHeight < 0) nh = height;
	else 		   nh = (newHeight * UpScale) / DownScale;

	if (nw < 0) nw = width;
	if (nh < 0) nh = height;

	if (nw > width) nw = width;
	if (nh > height) nh = height;

	if (!Quiet) Print("-> %s (%dx%dx%d)\n",out_fname,nw,nh,depth);

	out = fopen(out_fname,"wb");
	if (! out) {
	   Print("Cannot open '%s' for writing\n",out_fname);
	   exit(1);
	   }

	if (! make_fits_header(out, nw, nh, depth, bscale, bzero)) {
	   Print("Error writing FITS header\n");
	   exit(1);
	   }

	y1 = (height - nh)/2; y2 = y1 + nh-1;
	x1 = (width - nw)/2; x2 = x1 + nw-1;
	   
	// Convert the buffer back to original format
	if (depth == 8) {
	   unsigned char *ptr = (unsigned char *) (outbuffer + y1 * width + x1);

	   if (HistoProtect) {
	      unsigned int val = 255 * White / 100;
	      if (val > 255) val=255;
	      *ptr = 0; *(ptr+1) = val;
	      }

	   ptr = outbuffer; 

	   i=width * height;
	   while(i--) {
	   	//if (bscale != 1.0) *ptr /= bscale;
		if (bzero) *ptr -= bzero;
		}
	   }
	else if (depth == 16) {
	   char *tmp;
	   unsigned char *ptr = outbuffer;
	   unsigned short *iptr = (unsigned short *) (outbuffer + (y1 * width + x1) * bpp);

	   // Set pixel0 to black, pixel1 to white to prevent scaling
	   if (HistoProtect) {
	      	unsigned int val = 65535 * White / 100;
	      	if (val > 65535) val=65535;
	   	*iptr = 0; *(iptr+1) = val;
		}

	   iptr = (unsigned short *) outbuffer;
	   
	   i=width * height;

	   // byteswap
	   while(i--) {
	   	//if (bscale != 1.0) *iptr /= bscale;
		if (bzero) *iptr -= bzero;
		x=*ptr; *ptr = *(ptr+1); *(ptr+1) = x;
		iptr++; ptr += 2;
		}
	   }


	if (depth == 8) {
	   for(sptr = outbuffer + y1 * width + x1, y=y1; y<=y2; sptr += width, ++y)
		if (fwrite(sptr,x2-x1+1,1,out) != 1) {
		   Print("Short write on output to %s\n",out_fname);
		   exit(1);
		   }
	   }
	else if (depth == 16) {
	   sptr = outbuffer + (y1 * width + x1) * bpp;
	   for(y=y1; y<=y2; sptr += width*bpp, ++y)
		if (fwrite(sptr,(x2-x1+1)*bpp,1,out) != 1) {
		   Print("Short write on output to %s\n",out_fname);
		   exit(1);
		   }
	   }

	fclose(out);
	return(1);
	}
