#include "ppmcentre.h"

static void do_HNoiseFilter_window(unsigned char *buffer, int w, int h, int depth, int x1, int y1, int x2, int y2);
static int  CalculateCutout(unsigned char *in, int width, int height, int depth, int *X1, int *Y1, int *X2, int *Y2);
static int CutOut(unsigned char *in, unsigned char *out, int width, int height, int bpp, int x1, int y1, int x2, int y2);

static int
FindCentre(unsigned char *in,int width, int height, int depth, int *x_avg, int *y_avg)
	{
	int x,y,bpp;
	int count=0;	// count of significant pixels
	int x_total=0, y_total=0,size;
	unsigned char  *uptr = (unsigned char  *)in;
	unsigned short *iptr = (unsigned short *)in;
	int r,g,b,w,rowcount;
	double l,RealThreshHold;

        if (depth == 24) {
                RealThreshHold = ThreshHold;    // 8 bit data
                bpp = 3;
                }
        else if (depth == 16) {
                RealThreshHold = ThreshHold * 256;              // 16 bit data
                bpp = 2;
                }
        else if (depth == 8)  {
                bpp = 1;
                RealThreshHold = ThreshHold;    // 8 bit data
                }
        else {
                Print("FindCentre: Unsupported depth: %d\n",depth);
                exit(1);
                }

        for(y=0; y<height; ++y) {
           if (bpp == 3) for (x=rowcount=0; x<width; ++x) {
                b = *(uptr++); g = *(uptr++); r = *(uptr++);
                l = 0.299 * (double)r + 0.587 * (double)g + 0.114 * (double) b;
                if (l >= RealThreshHold) {
		   w = l/RealThreshHold;
		   x_total += x*w; y_total += y*w; count+=w; rowcount++; 
		   }
                }
           else if (bpp == 2) for (x=rowcount=0; x<width; ++x,++iptr) {
                if (*iptr >= RealThreshHold) {
		   w = *iptr/RealThreshHold;
                   x_total += x*w; y_total += y*w; count+=w; rowcount++; 
		   }
                }
           else if (bpp == 1) for (x=rowcount=0; x<width; ++x,++uptr) {
                if (*uptr >= RealThreshHold) {
		   w = *uptr/RealThreshHold;
		   x_total += x*w; y_total += y*w; count+=w; rowcount++; 
		   }
                }

           if (ForceProcess == 0 && (y==0 || y==height-1) && rowcount > 100) {
                Print("Warning: image data found near edge (row %d, %d pixels). Procesing cancelled\n",
                        y,rowcount);
                exit(1);
                }
           }

        if (count == 0 && ForceProcess==0) {
           Print("[no image] ");
           return(0);
           }

        if (count < MinPixels && ForceProcess==0) {
           Print("[Not enough pixels. Found %d, require %d] ",count,MinPixels);
           return(0);
           }

        *x_avg = x_total / count;
        *y_avg = y_total / count;

        if (!Quiet) Print("centre=(%d,%d), %d px\n",*x_avg,*y_avg,count);

	return(1);
	}

static int
CalculateCutout(unsigned char *in, int width, int height, int depth, int *X1, int *Y1, int *X2, int *Y2)
	{
	int x_avg,y_avg;
	int x1,y1,x2,y2;

	*X1=*Y1=*X2=*Y2=0;

	// Locate the centre of the object in the frame
	if (! FindCentre(in,width,height,depth,&x_avg,&y_avg))
	   return(0);

	// calculate the cutout from the source image (x1,y1) - (x2,y2).
	x1 = x_avg - CutX/2;
	x2 = x1 + CutX - 1;
	y1 = y_avg - CutY/2;
	y2 = y1 + CutY - 1;

	// make sure it stays symmetric around the located object when we have to clip
	while(1)
	   if (x1 < 0) {
		// cutout stays centered on target
		x2 += x1;
		x1 = 0;
		}
	   else if (x2 >= width) {
		// cutout stays centered on target
		x1 += (x2 - width + 1);
		x2 = width-1;
		}
	   else break;

	// Make sure y1,y2 stays centered on the target if we have to clip
	while(1)
	   if (y1<0) {
		y2 += y1;
		y1=0;
		}
	   else if (y2 >= height) {
		y1 += (y2 - height + 1);
		y2 = height-1;
		}
	   else break;

	if (x1<0 || y1<0 || x2>=width || y2 >=height) {
	   Print("calculateCutout: - Cannot calculate valid cutout\n");
	   exit(1);
	   }

	*X1 = x1; *X2 = x2; *Y1 = y1; *Y2 = y2;
	return(1);
	}

// Cutout the rectangle (x1,y1)-(x2,y2) and move it to the centre of the destination.
// Assume no resizing, ie both *in and *out have width and height as given
static int
CutOut(unsigned char *in, unsigned char *out, int width, int height, int bpp, int x1, int y1, int x2, int y2)
	{
	int xc=width/2;
	int yc=height/2;
	int y,i,cutx,cuty,yo,rowbytes;
	int X1,X2,Y1,Y2;  // output 
	unsigned char *src;

	// calculate new cutout, after possible clipping
	cutx = x2 - x1 + 1;
	cuty = y2 - y1 + 1;

	// Calculate where our cutout will end up so we can clip it
	// and adjust the src boundaries if required
	X1 = xc - cutx/2; X2 = X1 + cutx - 1;
	Y1 = yc - cuty/2; Y2 = Y1 + cuty - 1;

	if (X1<0) {x1 -= X1; X1=0;}
	if (Y1<0) {y1 -= Y1; Y1=0;}
	if (X2>=width)  {x2 -= (X2-width+1);  X2=width-1;}
	if (Y2>=height) {y2 -= (Y2-height+1); Y2=height-1;}

	// Now (x1,y1)-(x2,y2) is guaranteed to fit in the destination
	// recalculate the cutout size in case teh bounds have changed
	cutx = x2 - x1 + 1;
	cuty = y2 - y1 + 1;

	// zero rows before the cutout
	for(yo=0; yo<Y1; ++yo)
	   for(i=width*bpp; i>0; --i) *(out++) = 0;

	rowbytes = cutx * bpp;

	for(y = y1; y<=y2; ++y,++yo) {
	   
	   // zero bytes to the left of cutout
	   i=X1 * bpp; while(i-- > 0) *(out++) = 0;
	   
	   src = in + (y * width + x1) * bpp;

	   if (ChangeGamma) gammacpy(out,src,cutx,bpp,Gamma);
	   else memcpy(out,src,rowbytes);
	   out += rowbytes;

	   // zero bytes after cutout to end of row
	   i = (width - X1)*bpp - rowbytes;
	   while(i-- > 0) *(out++)=0;
	   }

	// zero rows after the cutout
	while(yo++ < height) for(i=width*bpp; i>0; --i) *(out++) = 0;

	return(1);
	}

// If we rescale then reset width and height
int process(int *width, int *height, int depth, unsigned char *in, unsigned char *out, char *outfname)
	{
	int bpp=depth/8;
	int x1,x2,y1,y2;
	int i,CentreFocus=1;

	// Locate the object and create the cutout rectangle from (x1,y1) to (x2,y2)
	if (DoCutout) {
	   if (! CalculateCutout(in,*width,*height,depth,&x1,&y1,&x2,&y2))
	      return(0);
	   }
	else {
	   x1=0; y1=0; x2=*width-1; y2=*height-1;
	   }

	// Apply the Horizontal Noise filter
	if (HNoiseFilter) {
	   if (!Quiet) Print("Hnoise: ");
	   do_HNoiseFilter(in,*width,*height,depth,x1,y1,x2,y2);
	   if (!Quiet) Print("\n");
	   }

	// Apply input filter if required
	if (InputFilter && !Quiet) Print("IFilter: ");
        for(i=0; i<InputFilter; ++i) {
           int count = input_filter(in,*width,*height,depth,x1,y1,x2,y2);
	   if (!Quiet) {
		if (i) Print("+");
		Print("%d",count);
		}
	   }
	if (InputFilter && !Quiet) Print("\n");

	if (QEstimator) {
	   int X1=x1,Y1=y1,X2=x2,Y2=y2;

	   if (CentreFocus) {
		// Only quality estimate the centre half of the image
	   	int w=(x2-x1+1)/2,h=(y2-y1+1)/2;
	   	int xc=(x1+x2)/2;
	   	int yc=(y1+y2)/2;

	 	X1 = xc-(w-1)/2; X2=X1+w-1;
	 	Y2 = yc-(h-1)/2; Y2=Y1+h-1;
		}

	   if (!Quiet) Print("Quality: ");
	   double q = QualityEstimate(in,*width,*height,depth,X1,Y1,X2,Y2);
	   if (!Quiet) Print(" = %-2.2lf\n",q);
	   QAssociate(outfname,q);
	   }

	if (DoCutout)
	   CutOut(in,out,*width,*height,bpp,x1,y1,x2,y2);
	else
	   memcpy(out,in,*width * *height * bpp);

	// If we have to upscale the data then do it from 
	// bottom -> top so we can just copy the existing 
	// image data in place.
	if (UpScale > 1) {
	   upscale_image(out,width,height,bpp,UpScale);

	  if (UpScale_Smoothing)
	     smooth_image(out,*width,*height,depth,UpScale);
	  }

	if (DownScale > 1)
	   downscale_image(out,width,height,bpp,DownScale);

	return(1);
	}

int
gammacpy(unsigned char *dst, unsigned char *src, int npix, int bpp, double gamma)
	{
	static double last_gamma = 1.0;
	static int last_bpp = -1;
	static unsigned char *table = NULL;
	static unsigned short *itable;
	unsigned short *idst = (unsigned short *) dst;
	unsigned short *isrc = (unsigned short *) src;

	// Do we have to init the lookup table?
	if (last_bpp != bpp || last_gamma != gamma || table == NULL) {
	   if (table) {
		Print("\nfree table\n");
		free(table);
		}

	   Print("\nCreating gamma table %f for %dbpp\n",gamma,bpp);

	   switch(bpp) {
		case 1:
		case 3:
		table = (unsigned char *) malloc(256); if (! table) {
		   Print("adjust_gamma: Out of memory\n");
		   exit(1);
		   }
		create_gamma_table_8bpp(table,gamma);
		break;

		case 2:
		table = (unsigned char *) malloc(65536 * sizeof(short)); if (! table) {
		   Print("adjust_gamma: Out of memory\n");
		   exit(1);
		   }
		itable = (unsigned short *) table;
		create_gamma_table_16bpp(itable,gamma);
		break;

		default:
		Print("adjust_gamma: unknown bpp %d\n",bpp);
		exit(1);
		}
	   }

	switch (bpp) {
	   	case 1:
		while(npix--) *(dst++) = table[*(src++)];
		break;

		case 2:
		while(npix--) *(idst++) = itable[*(isrc++)];
		break;

		case 3:
		while(npix--) {
		   *(dst++) = table[*(src++)];
		   *(dst++) = table[*(src++)];
		   *(dst++) = table[*(src++)];
		   }
		break;
		}

	last_gamma = gamma;
	last_bpp = bpp;
	return(1);
	}

int
create_gamma_table_8bpp(unsigned char *table, double gamma)
	{
	int i;

	for(i=0; i<256; ++i) {
	   int v = pow((double)i / 255.0, 1.0/gamma) * 255;
	   if (v>255) v=255;
	   table[i] = v;
	   }

	return(1);
	}

int
create_gamma_table_16bpp(unsigned short *table, double gamma)
	{
	int i;

	for(i=0; i<65536; ++i) {
	   double v = pow((double)i / 65535.0, 1.0/gamma) * 65535;
	   if (v>65535) v=65535;
	   table[i] = v;
	   }

	return(1);
	}

void
do_HNoiseFilter(unsigned char *buffer, int w, int h, int depth, int x1, int y1, int x2, int y2)
	{
	int window=50;

	// Process the image in slices to focus on transient noise
	while(x1 < x2) {
	   do_HNoiseFilter_window(buffer, w, h, depth, x1, y1, x1+window-1, y2);
	   x1 += window;
	   }
	}
	   
// Force (x1,y1) and (x2,y2) to be a rectangle within (0,0) - (w-1,h-1)
int Clip(int w, int h, int *x1, int *y1, int *x2, int *y2)
	{
	if (*x2 < *x1 || *y2 < *y1) {
	   Print("\n\nClipBouondaries: (%d,%d) - (%d,%d) invalid region\n",*x1,*y1,*x2,*y2);
	   exit(1);
	   }

	if (*x1<0) *x1=0; if (*x1>=w) *x1=w-1;
	if (*x2<0) *x2=0; if (*x2>=w) *x2=w-1;

	if (*y1<0) *y1=0; if (*y1>=h) *y1=h-1;
	if (*y2<0) *y2=0; if (*y2>=h) *y2=h-1;

	return(1);
	}

static void
do_HNoiseFilter_window(unsigned char *buffer, int w, int h, int depth, int x1, int y1, int x2, int y2)
	{
	int i,nrows,y,r,count;
	int pixels_per_row;
	int row,diff;
	unsigned short *ubuffer = (unsigned short *)buffer;
	double *row_avg,*diff_avg;
	int minval = 50;
	int uminval = minval*256;
	int threshold = 3;
	int *temp_buffer;
	int *modified_row;
	int useupper = 1;
	int uselower = 1;

	Clip(w,h,&x1,&y1,&x2,&y2);

	pixels_per_row = x2-x1+1;
	nrows = y2-y1+1;
	row_avg = (double *)malloc(sizeof(double) * nrows);
	diff_avg = (double *)malloc(sizeof(double) * nrows);
	if (row_avg == NULL || diff_avg == NULL) {
	   Print("Out of memory\n");
	   exit(1);
	   }

	modified_row = (int *)malloc(sizeof(int) * nrows);
	temp_buffer = (int *)malloc(sizeof(int) * pixels_per_row * nrows);
	if (temp_buffer == NULL) {
	   Print("Out of memory\n");
	   exit(1);
	   }

	switch(depth) {
	   case 16:
		for(r=0,y=y1; y<=y2; ++y,++r) {
		   int o = y*w + x1;
		   row=diff=count=0;
		   for(i=0; i<pixels_per_row; ++i,++o)
			if (ubuffer[o] >= uminval) {
			   int v = (int)ubuffer[o]>>8;
			   row += v; diff += v;
			   diff -= (int)ubuffer[o-w]>>8;
			   count++;
			   }
		   if (count<10) row_avg[r]=-1;
		   else {
			row_avg[r] = (double)row / (double)count;
			diff_avg[r] = (double)diff / (double)count;
			diff_avg[r] /= row_avg[r]; diff_avg[r]*=100;
			}
		   }
		break;

	   default:
		Print("Bit depth not handled\n");
		exit(1);
		}

	// Any row that is significantly brighter than its previous and following rows is
	// suspicious
	count=0;
	for(i=0; i<nrows; ++i) modified_row[i]=0;

	for(r=1; r<nrows-1; ++r) 
	   if (row_avg[r-1]>0 && row_avg[r]>0 && row_avg[r+1]>0) {
	   	int d1 = diff_avg[r] - diff_avg[r-1];
	   	int d2 = diff_avg[r] - diff_avg[r+1];

	   	if ((uselower && d1<=-threshold && d2<=-threshold) 
		   || (useupper && d1>=threshold && d2>=threshold)) {
		  double br = ((row_avg[r]/row_avg[r-1]) + (row_avg[r]/row_avg[r+1]))/2;
		  int o_dst = r * pixels_per_row;
		  int o_src = (y1+r) * w + x1;

		  for(i=0; i<pixels_per_row; ++i,++o_src,++o_dst)
		   	   temp_buffer[o_dst] = (
				(ubuffer[o_src] / br) * 4 + 
				ubuffer[o_src-w] + 
			   	ubuffer[o_src+w]) / 6;

		  modified_row[r] = 1;
		  ++count;
		  }
		}

	if (!Quiet) Print("%d ",count);

	// Copy back the modified rows
	for(r=1; r<nrows-1 && count>0; ++r) if (modified_row[r]) {
	   int o_src = r * pixels_per_row;
	   int o_dst = (y1+r) * w + x1;
	   for(i=0; i<pixels_per_row; ++i,++o_src,++o_dst) ubuffer[o_dst] = temp_buffer[o_src];
	   --count;
	   }

	free(temp_buffer);
	free(row_avg);
	free(diff_avg);
	free(modified_row);
	}
