#include "ppmcentre.h"

static double *fdata = NULL;  // This is our buffer
static int NPixels=0;
static int Width,Height,Depth;
static int Count=0;
static int Bpp=0;

int
stack_frame(unsigned char *data, int width, int height, int depth)
	{
	int i;
	unsigned short *sdata = (unsigned short *)data;
	double v;

	if (fdata == NULL) {
	   // First frame, allocate buffer
	   Width = width; Height=height; Depth = depth;
	   Bpp = Depth/8;

	   NPixels = width * height;
	   fdata = (double *)malloc(sizeof(double) * NPixels);
	   if (fdata==NULL) {
		Print("stack_frame: out of memory\n");
		exit(1);
		}

	   if (Depth != 16) {
		Print("-stack only supported for 16bpp monochrome data\n");
		exit(1);
		}

	   // Copy the first frame into the buffer
	   for(i=0; i<NPixels; ++i)
		fdata[i] = sdata[i];

	   // init the frame counter
	   Count = 1;

	   return Count;
	   }

	// Else we are adding a frame to the existing stack
	for(i=0; i<NPixels; ++i) {
	   v = sdata[i];
	   fdata[i] += v;
	   }
	
	return ++Count;
	}

static int
fetch_stack_u16(unsigned short *data)
	{
	int i;
	int v;

	if (! Count) {
	   Print("fetch_stack_u16: no frames stacked\n");
	   return 0;
	   }

	Print("Averaging %d frames\n",Count);
	for(i=0; i<NPixels; ++i) {
	   v = fdata[i]/Count;
	   if (v>65535) {
		v=65535;
		Print("Warning: truncated data\n");
	 	}
	   data[i] = v;
	   }
	return Count;
	}

int
write_stack_file(char *fname)
	{
	FILE *out;
	double bscale = 1.0;
	int bzero = 0;
	int i,x,min,max;
	unsigned short *iptr;
	unsigned char *data,*ptr;

	// Allocate the buffer
	data = (unsigned char *) malloc(Width * Height * Depth/8);
	if (data == NULL) {
	   Print("write_stack_file: Out of Memory\n");
	   exit(1);
	   }

	// Fetch the image
	iptr = (unsigned short *)data;
	fetch_stack_u16(iptr);
	
	if (Depth == 16) {
	   bscale=1.0;
	   bzero = 32768;
	   }

        // Set pixel0 to black, pixel1 to white to prevent scaling
        if (HistoProtect) {
             unsigned int val = 65535 * White / 100;
             if (val > 65535) val=65535;
             *iptr = 0; *(iptr+1) = val;
             }

	// Byteswap the data
        i=Width * Height;
	ptr = data;
	min=65535; max=0;
        while(i--) {
	     if (*iptr<min) min=*iptr; if (*iptr>max) max=*iptr;
             if (bzero) *iptr -= bzero;
             x=*ptr; *ptr = *(ptr+1); *(ptr+1) = x;
             iptr++; ptr += 2;
             }

	Print("[stackfile min=%d max=%d] ",min,max);

	out = fopen(fname,"w");
	if (!out) {
	   Print("write_stack_file: Error opening output file '%s'\n",fname);
	   exit(1);
	   }

        if (! make_fits_header(out, Width, Height, Depth, bscale, bzero)) {
           Print("Error writing FITS header\n");
           exit(1);
           }
	
        if (fwrite(data,Width*Height*Bpp,1,out) != 1) {
              Print("Short write on output to %s\n",fname);
              exit(1);
              }

	fclose(out);
	free(data);
	return(1);
	}

static int MWidth=0,MHeight=0,MDepth=0,MBpp;
static int Mbzero,MBufsize;
static double Mbscale;
static double MAvg;
static unsigned char *MBuffer=NULL;

static int
load_mergefile(char *fname)
	{
	FILE *in;
	int i,x,count=0;
	double avg;
	int minval8 = ThreshHold;
	int minval16 = minval8<<8 ; // don't process the background

	in = fopen(fname,"r");
	if (!in) {
	   Print("load_mergefile: cannot open '%s' for reading\n",fname);
	   exit(1);
	   }

	read_fits_header(in,&MWidth,&MHeight,&MDepth,&Mbscale,&Mbzero);
	if (!Quiet)
	   Print("[Merge: %dx%dx%d bzero=%d, bscale=%f] ",MWidth,MHeight,MDepth,Mbzero,Mbscale);

	MBpp = MDepth/8;

        /*
         * Allocate the buffer and load the data
         */
        MBufsize = MWidth * MHeight * MBpp;

        MBuffer = (unsigned char *)malloc(MBufsize);
        if (MBuffer == NULL) {
             Print("load_mergefile: Cannot malloc buffer of %d bytes\n",MBufsize);
             exit(1);
             }

        if (fread(MBuffer,MBufsize,1,in) != 1) {
              Print("load_mergefile: Error in reading file data for %s\n",fname);
              exit(1);
              }
	fclose(in);

       // Undo the endian-ness of the data for 16bpp FITS data
        if (MDepth == 16) {
           unsigned char *tmp,*ptr;
           unsigned short *iptr;

           ptr = MBuffer; iptr = (unsigned short *)ptr;
           i=MWidth * MHeight;
	   count=0; avg=0;

           while(i--) {
                x = *ptr; *ptr = *(ptr+1); *(ptr+1) = x;
                if (Mbzero) *iptr += Mbzero;
                //if (Mbscale != 1.0) *iptr *= Mbscale;
		if (*iptr >= minval16) {avg += *iptr; ++count;}
                ptr += 2; iptr++;
                }
           }
        else if (MDepth == 8) {
           // Unsigned 8 bit data
           unsigned char *ptr = MBuffer;
           i = MWidth * MHeight;
           while(i--) {
                if (Mbzero) *ptr += Mbzero;
                //if (Mbscale != 1.0) *ptr *= Mbscale;
		if (*ptr >= minval8) {avg += *ptr; ++count; }
                ptr++;
                }
           }
        else {
           Print("load_mergefile: Unsupported depth: %d\n",MDepth);
           exit(1);
           }

	if (count < MinPixels) {
	   Print("LoadMerge: found no pixels\n");
	   exit(0);
	   }

	MAvg = (double)avg / count;
	Print("[LoadMerge%d avg=%lf] ",MDepth,MAvg);

	return(1);
	}

#define MERGE_EQ(a,b) (((a) + ((a)>>1) + ((b)>>1))>>1)

static void
merge_16_16(unsigned short *src, unsigned short *ref, int npix)
	{
	int i,count,high_count=0,low_count=0;
	int minval = ThreshHold << 8;
	double avg,d,scale;

	// calculate average brightness for scaling purposes
	for(i=avg=count=0; i<npix; ++i)
	   if (src[i] >= minval) {avg += src[i]; ++count;}
		
	if (count < MinPixels) {
	   Print("merge_data: found no significant pixels\n");
	   exit(1);
	   }

	avg /= (double) count;
	scale = MAvg/avg;

	for(i=0; i<npix; ++i,++src,++ref) {
	   int s = (double)*src * scale + 0.5;
	   if (s >= minval && *ref) {
		double d = s - *ref; d /= *ref;

		if (d >= MergeThreshHold) {
		   *src = MERGE_EQ(s,*ref); ++high_count; }
		else if (d <= -MergeThreshHold) {
		   *src = MERGE_EQ(s,*ref); ++low_count; }
		}
	   }

	Print("%d/%d pixels] ",high_count,low_count);
	return;
	}

static void
merge_8_8(unsigned char *src, unsigned char *ref, int npix)
	{
	int i,count,high_count=0,low_count=0;
	int minval = ThreshHold;
	double avg,d,scale;

	// calculate average brightness for scaling purposes
	for(i=avg=count=0; i<npix; ++i)
	   if (src[i] >= minval) {avg += src[i]; ++count;}
		
	if (count < MinPixels) {
	   Print("merge_data: found no significant pixels\n");
	   exit(1);
	   }

	avg /= (double) count;
	scale = MAvg/avg;

	for(i=0; i<npix; ++i,++src,++ref) {
	   int s = (double)*src * scale + 0.5;
	   if (s >= minval && *ref) {
		double d = s - *ref; d /= *ref;

		if (d >= MergeThreshHold) {
		   *src = MERGE_EQ(s,*ref); ++high_count; }
		else if (d <= -MergeThreshHold) {
		   *src = MERGE_EQ(s,*ref); ++low_count; }
		}
	   }

	Print("%d/%d pixels] ",high_count,low_count);
	return;
	}

static void
merge_16_8(unsigned short *src, unsigned char *ref, int npix)
	{
	int i,count,high_count=0,low_count=0;
	int minval = ThreshHold << 8;
	double avg,d,scale;

	// calculate average brightness for scaling purposes
	for(i=avg=count=0; i<npix; ++i)
	   if (src[i] >= minval) {avg += src[i]>>8; ++count;}
		
	if (count < MinPixels) {
	   Print("merge_data: found no significant pixels\n");
	   exit(1);
	   }

	avg /= (double) count;
	scale = MAvg/avg;

	for(i=0; i<npix; ++i,++src,++ref) {
	   int s = (double)*src * scale + 0.5;
	   int r = *ref << 8;

	   if (s >= minval && r) {
		double d = (s - r) / r;

		if (d >= MergeThreshHold) {
		   *src = MERGE_EQ(s,r); ++high_count; }
		else if (d <= -MergeThreshHold) {
		   *src = MERGE_EQ(s,r); ++low_count; }
		}
	   }

	Print("avg=%lf scale=%lf %d/%d pixels] ",avg,scale,high_count,low_count);
	return;
	}

static void
merge_8_16(unsigned char *src, unsigned short *ref, int npix)
	{
	int i,count,high_count=0,low_count=0;
	int minval = ThreshHold;
	double avg,d,scale;

	// calculate average brightness for scaling purposes
	for(i=avg=count=0; i<npix; ++i)
	   if (src[i] >= minval) {avg += src[i]<<8; ++count;}
		
	if (count < MinPixels) {
	   Print("merge_data: found no significant pixels\n");
	   exit(1);
	   }

	avg /= (double) count;
	scale = MAvg/avg;

	for(i=0; i<npix; ++i,++src,++ref) {
	   int s = (double)*src * scale + 0.5;
	   int r = *ref >> 8;

	   if (s >= minval && r) {
		double d = (s - r) / r;

		if (d >= MergeThreshHold) {
		   *src = MERGE_EQ(s,r); ++high_count; }
		else if (d <= -MergeThreshHold) {
		   *src = MERGE_EQ(s,r); ++low_count; }
		}
	   }

	Print("%d/%d pixels] ",high_count,low_count);
	return;
	}

int
merge_data(char *mfile, unsigned char *data, int width, int height, int depth)
	{
	int npix;

	if (MBuffer == NULL) 
	   load_mergefile(mfile);

	if (MBuffer==NULL || (width != MWidth) || (MHeight != height)) {
	   Print("merge_data: Frame size does not match mergefile\n");
	   exit(1);
	   }

	npix = width * height;
	Print("[merge %d+%d, ",depth,MDepth);
	if (depth==16) {
	   if (MDepth==16)
		merge_16_16((unsigned short *)data, (unsigned short *)MBuffer,npix);
	   else if (MDepth==8)
		merge_16_8((unsigned short *)data, MBuffer,npix);
	   }
	else if (depth==8) {
	   if (MDepth==16)
		merge_8_16(data, (unsigned short *)MBuffer,npix);
	   else if (MDepth==8)
		merge_8_8(data, MBuffer,npix);
	   }

	return(1);
	}
